<?php

namespace Drupal\tampers_extra\Plugin\Tamper;

use Drupal\Core\Form\FormStateInterface;
use Drupal\tamper\TamperBase;
use Drupal\tamper\TamperableItemInterface;

/**
 * Adds VAT on top of an incoming price.
 *
 * @Tamper(
 *   id = "price_with_vat",
 *   label = @Translation("Calculate price with VAT"),
 *   description = @Translation("Applies a VAT amount using a feed field containing the VAT rate."),
 *   category = @Translation("Commerce")
 * )
 */
class PriceWithVat extends TamperBase
{

    /**
     * Configuration key for the VAT rate field.
     */
    public const VAT_RATE_FIELD = 'vat_rate_field';

    /**
     * Configuration key for the rate type.
     */
    public const RATE_TYPE = 'rate_type';

    /**
     * Configuration key for the fallback rate.
     */
    public const FALLBACK_RATE = 'fallback_rate';

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return parent::defaultConfiguration() + [
            self::VAT_RATE_FIELD => '',
            self::RATE_TYPE => 'decimal',
            self::FALLBACK_RATE => '',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $options = $this->getSourceOptions();

        $form[self::VAT_RATE_FIELD] = [
            '#type' => $options ? 'select' : 'textfield',
            '#title' => $this->t('VAT rate field'),
            '#description' => $this->t("Feed column that contains the VAT amount."),
            '#default_value' => $this->getSetting(self::VAT_RATE_FIELD),
            '#required' => TRUE,
        ];

        if ($options) {
            $form[self::VAT_RATE_FIELD]['#options'] = $options;
            $form[self::VAT_RATE_FIELD]['#empty_option'] = $this->t('- Select -');
        }

        $form[self::RATE_TYPE] = [
            '#type' => 'radios',
            '#title' => $this->t('Rate type'),
            '#description' => $this->t('Choose how the VAT rate value should be interpreted.'),
            '#default_value' => $this->getSetting(self::RATE_TYPE),
            '#options' => [
                'decimal' => $this->t('Decimal (e.g. 0.18 for 18%)'),
                'percentage' => $this->t('Percentage (e.g. 18 for 18%)'),
            ],
            '#required' => TRUE,
        ];

        $form[self::FALLBACK_RATE] = [
            '#type' => 'textfield',
            '#title' => $this->t('Fallback Rate'),
            '#description' => $this->t('Optional fallback rate to use if the source field is empty (e.g. 20 for 20% or 0.20 for 20%). Follows the Rate Type setting.'),
            '#default_value' => $this->getSetting(self::FALLBACK_RATE),
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        $this->setConfiguration([
            self::VAT_RATE_FIELD => trim((string) $form_state->getValue(self::VAT_RATE_FIELD)),
            self::RATE_TYPE => $form_state->getValue(self::RATE_TYPE) === 'percentage' ? 'percentage' : 'decimal',
            self::FALLBACK_RATE => trim((string) $form_state->getValue(self::FALLBACK_RATE)),
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function tamper($data, ?TamperableItemInterface $item = NULL)
    {
        if (!$item instanceof TamperableItemInterface) {
            return $data;
        }

        $vat_field = $this->getSetting(self::VAT_RATE_FIELD);
        if ($vat_field === '') {
            return $data;
        }

        $vat_value = $item->getSourceProperty($vat_field);

        // Use fallback if value is missing/invalid or ZERO
        if (!is_numeric($vat_value) || (float)$vat_value == 0) {
            $fallback = $this->getSetting(self::FALLBACK_RATE);

            if (is_numeric($fallback)) {
                $vat_value = $fallback;
            } else {
                // If value was 0 and no fallback, we keep it as 0 (valid 0% VAT)
                if (is_numeric($vat_value) && (float)$vat_value == 0) {
                    // VAT is 0 and no fallback set. Using 0%.
                } else {
                    return $data;
                }
            }
        }

        if (!is_numeric($data)) {
            return $data;
        }

        $rate = (float) $vat_value;
        if ($this->getSetting(self::RATE_TYPE) === 'percentage') {
            $rate = $rate / 100;
        }

        $result = (float) $data * (1 + $rate);

        $result = number_format($result, 2, '.', '');

        return number_format($result, 6, '.', '');
    }

    /**
     * Builds a list of selectable source options from the feed mapping.
     */
    protected function getSourceOptions(): array
    {
        if (!$this->sourceDefinition) {
            return [];
        }

        $options = $this->sourceDefinition->getList();
        if (!is_array($options) || $options === []) {
            return [];
        }

        $keys = array_keys($options);
        $is_associative = $keys !== range(0, count($keys) - 1);

        if (!$is_associative) {
            $normalized = [];
            foreach ($options as $value) {
                if (!is_scalar($value)) {
                    continue;
                }
                $value = trim((string) $value);
                if ($value === '') {
                    continue;
                }
                $normalized[$value] = $value;
            }
            return $normalized;
        }

        $normalized = [];
        foreach ($options as $key => $label) {
            if (!is_scalar($key)) {
                continue;
            }
            $key_string = trim((string) $key);
            if ($key_string === '') {
                continue;
            }
            $normalized[$key_string] = is_scalar($label) ? (string) $label : $key_string;
        }

        return $normalized;
    }
}
