<?php

namespace Drupal\tampers_extra\Plugin\Tamper;

use Drupal\Core\Form\FormStateInterface;
use Drupal\tamper\TamperBase;
use Drupal\tamper\TamperableItemInterface;

/**
 * Converts a supplier price using another row's exchange rate fields.
 *
 * @Tamper(
 *   id = "supplier_price_from_exchange_rate",
 *   label = @Translation("Convert supplier price using exchange rate"),
 *   description = @Translation("Derives the target currency price by calculating an exchange rate from two other feed columns and multiplying the current value."),
 *   category = @Translation("Commerce")
 * )
 */
class PriceCalculator extends TamperBase
{

    /**
     * Configuration key for the source-currency column.
     */
    public const RATE_SOURCE_FIELD = 'rate_source_field';

    /**
     * Configuration key for the target-currency column.
     */
    public const RATE_LOCAL_FIELD = 'rate_local_field';

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return parent::defaultConfiguration() + [
            self::RATE_SOURCE_FIELD => '',
            self::RATE_LOCAL_FIELD => '',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $options = $this->getSourceOptions();

        $form[self::RATE_SOURCE_FIELD] = [
            '#type' => $options ? 'select' : 'textfield',
            '#title' => $this->t('Source price field'),
            '#description' => $this->t("Machine name of the source column that stores the supplier's original currency price (for example 'price_foreign')."),
            '#default_value' => $this->getSetting(self::RATE_SOURCE_FIELD),
            '#required' => TRUE,
        ];

        if ($options) {
            $form[self::RATE_SOURCE_FIELD]['#options'] = $options;
            $form[self::RATE_SOURCE_FIELD]['#empty_option'] = $this->t('- Select -');
        }

        $form[self::RATE_LOCAL_FIELD] = [
            '#type' => $options ? 'select' : 'textfield',
            '#title' => $this->t('Local price field'),
            '#description' => $this->t("Machine name of the source column that stores the target currency price (for example 'price_try')."),
            '#default_value' => $this->getSetting(self::RATE_LOCAL_FIELD),
            '#required' => TRUE,
        ];

        if ($options) {
            $form[self::RATE_LOCAL_FIELD]['#options'] = $options;
            $form[self::RATE_LOCAL_FIELD]['#empty_option'] = $this->t('- Select -');
        }

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        $this->setConfiguration([
            self::RATE_SOURCE_FIELD => trim((string) $form_state->getValue(self::RATE_SOURCE_FIELD)),
            self::RATE_LOCAL_FIELD => trim((string) $form_state->getValue(self::RATE_LOCAL_FIELD)),
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function tamper($data, ?TamperableItemInterface $item = NULL)
    {
        if (!$item instanceof TamperableItemInterface) {
            return $data;
        }

        $source_field = $this->getSetting(self::RATE_SOURCE_FIELD);
        $local_field = $this->getSetting(self::RATE_LOCAL_FIELD);

        if ($source_field === '' || $local_field === '') {
            return $data;
        }

        $source_price = $item->getSourceProperty($source_field);
        $local_price = $item->getSourceProperty($local_field);

        if (!is_numeric($source_price) || !is_numeric($local_price)) {
            return $data;
        }

        $source_price = (float) $source_price;
        $local_price = (float) $local_price;

        if ($source_price == 0.0) {
            return $data;
        }

        $exchange_rate = $local_price / $source_price;

        if (!is_numeric($data)) {
            return $data;
        }

        $converted = (float) $data * $exchange_rate;
        $result = number_format($converted, 2, '.', '');
        $result = number_format($converted, 6, '.', '');

        return $result;
    }

    /**
     * Builds a list of selectable source options from the feed mapping.
     */
    protected function getSourceOptions(): array
    {
        if (!$this->sourceDefinition) {
            return [];
        }

        $options = $this->sourceDefinition->getList();
        if (!is_array($options) || $options === []) {
            return [];
        }

        $keys = array_keys($options);
        $is_associative = $keys !== range(0, count($keys) - 1);

        if (!$is_associative) {
            // When the list is sequential ensure the option value matches the label so
            // selecting from the dropdown stores the expected source machine name.
            $normalized = [];
            foreach ($options as $value) {
                if (!is_scalar($value)) {
                    continue;
                }
                $value = trim((string) $value);
                if ($value === '') {
                    continue;
                }
                $normalized[$value] = $value;
            }
            return $normalized;
        }

        $normalized = [];
        foreach ($options as $key => $label) {
            if (!is_scalar($key)) {
                continue;
            }
            $key_string = trim((string) $key);
            if ($key_string === '') {
                continue;
            }
            $normalized[$key_string] = is_scalar($label) ? (string) $label : $key_string;
        }

        return $normalized;
    }
}
