<?php

namespace Drupal\tampers_extra\Plugin\Tamper;

use Drupal\Core\Form\FormStateInterface;
use Drupal\tamper\TamperableItemInterface;
use Drupal\tamper\TamperBase;

/**
 * @Tamper(
 *   id = "default_between",
 *   label = @Translation("Default between"),
 *   description = @Translation("Set a default value if the input is not within a given range."),
 *   category = "Number"
 * )
 */
class DefaultBetween extends TamperBase
{

  const SETTING_DEFAULT_VALUE = 'default_value';
  const SETTING_MIN_VALUE = 'min_value';
  const SETTING_MAX_VALUE = 'max_value';
  const SETTING_IGNORE_MIN = 'ignore_min';
  const SETTING_IGNORE_MAX = 'ignore_max';
  const SETTING_RETURN_TYPE = 'return_type';
  const SETTING_FALLBACK_SOURCE = 'fallback_source';
  const SETTING_MULTIPLY = 'multiply';

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array
  {
    return [
      self::SETTING_DEFAULT_VALUE => '',
      self::SETTING_MIN_VALUE => '',
      self::SETTING_MAX_VALUE => '',
      self::SETTING_IGNORE_MIN => FALSE,
      self::SETTING_IGNORE_MAX => FALSE,
      self::SETTING_RETURN_TYPE => 'default_value',
      self::SETTING_FALLBACK_SOURCE => '',
      self::SETTING_MULTIPLY => '1',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
  {
    $form[self::SETTING_MULTIPLY] = [
      '#type' => 'number',
      '#title' => $this->t('Multiply'),
      '#description' => $this->t('Multiply the value by this number before checking the range. Useful for percentages (e.g., 100 to convert 0.20 to 20).'),
      '#default_value' => $this->configuration[self::SETTING_MULTIPLY],
      '#step' => 'any',
    ];

    $form[self::SETTING_MIN_VALUE] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum value'),
      '#description' => $this->t('The minimum value for the range.'),
      '#default_value' => $this->configuration[self::SETTING_MIN_VALUE],
      '#step' => 'any',
      '#states' => [
        'visible' => [
          ':input[name$="[ignore_min]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form[self::SETTING_IGNORE_MIN] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Don't check minimum"),
      '#default_value' => $this->configuration[self::SETTING_IGNORE_MIN],
    ];

    $form[self::SETTING_MAX_VALUE] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum value'),
      '#description' => $this->t('The maximum value for the range.'),
      '#default_value' => $this->configuration[self::SETTING_MAX_VALUE],
      '#step' => 'any',
      '#states' => [
        'visible' => [
          ':input[name$="[ignore_max]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form[self::SETTING_IGNORE_MAX] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Don't check maximum"),
      '#default_value' => $this->configuration[self::SETTING_IGNORE_MAX],
    ];

    $form[self::SETTING_RETURN_TYPE] = [
      '#type' => 'radios',
      '#title' => $this->t('Return value when out of range'),
      '#description' => $this->t('Select what value to return when the input is not within the range.'),
      '#options' => [
        'default_value' => $this->t('Return default value'),
        'other_source' => $this->t('Return other source value'),
      ],
      '#default_value' => $this->configuration[self::SETTING_RETURN_TYPE],
    ];

    $form[self::SETTING_DEFAULT_VALUE] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default value'),
      '#description' => $this->t('The default value to set if the input is not within the range.'),
      '#default_value' => $this->configuration[self::SETTING_DEFAULT_VALUE],
      '#states' => [
        'visible' => [
          ':input[name$="[return_type]"]' => ['value' => 'default_value'],
        ],
      ],
    ];

    $source_options = $this->getSourceOptions();

    $form[self::SETTING_FALLBACK_SOURCE] = [
      '#type' => $source_options ? 'select' : 'textfield',
      '#title' => $this->t('Fallback source'),
      '#description' => $this->t('Select the source to use when the value is out of range.'),
      '#default_value' => $this->configuration[self::SETTING_FALLBACK_SOURCE],
      '#states' => [
        'visible' => [
          ':input[name$="[return_type]"]' => ['value' => 'other_source'],
        ],
        'required' => [
          ':input[name$="[return_type]"]' => ['value' => 'other_source'],
        ],
      ],
    ];

    if ($source_options) {
      $form[self::SETTING_FALLBACK_SOURCE]['#options'] = $source_options;
      $form[self::SETTING_FALLBACK_SOURCE]['#empty_option'] = $this->t('- Select a source -');
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
  {
    parent::submitConfigurationForm($form, $form_state);
    $this->setConfiguration([
      self::SETTING_DEFAULT_VALUE => $form_state->getValue(self::SETTING_DEFAULT_VALUE),
      self::SETTING_MIN_VALUE => $form_state->getValue(self::SETTING_MIN_VALUE),
      self::SETTING_MAX_VALUE => $form_state->getValue(self::SETTING_MAX_VALUE),
      self::SETTING_IGNORE_MIN => $form_state->getValue(self::SETTING_IGNORE_MIN),
      self::SETTING_IGNORE_MAX => $form_state->getValue(self::SETTING_IGNORE_MAX),
      self::SETTING_RETURN_TYPE => $form_state->getValue(self::SETTING_RETURN_TYPE),
      self::SETTING_FALLBACK_SOURCE => $form_state->getValue(self::SETTING_FALLBACK_SOURCE),
      self::SETTING_MULTIPLY => $form_state->getValue(self::SETTING_MULTIPLY),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function tamper($data, ?TamperableItemInterface $item = NULL): mixed
  {
    $default_value = $this->configuration[self::SETTING_DEFAULT_VALUE];
    $min = $this->configuration[self::SETTING_MIN_VALUE];
    $max = $this->configuration[self::SETTING_MAX_VALUE];
    $ignore_min = $this->configuration[self::SETTING_IGNORE_MIN];
    $ignore_max = $this->configuration[self::SETTING_IGNORE_MAX];
    $return_type = $this->configuration[self::SETTING_RETURN_TYPE] ?? 'default_value';
    $fallback_source = $this->configuration[self::SETTING_FALLBACK_SOURCE] ?? '';
    $multiply = $this->configuration[self::SETTING_MULTIPLY] ?? '1';

    // Cast to numeric if possible.
    $numeric_data = is_numeric($data) ? (float) $data : NULL;
    $numeric_min = is_numeric($min) ? (float) $min : NULL;
    $numeric_max = is_numeric($max) ? (float) $max : NULL;
    $numeric_multiply = is_numeric($multiply) ? (float) $multiply : 1;

    if ($numeric_data === NULL) {
      return $this->getFallbackValue($return_type, $default_value, $fallback_source, $item, $numeric_multiply);
    }

    // Apply multiplication before range check
    $multiplied_data = $numeric_data * $numeric_multiply;

    $is_less_than_min = !$ignore_min && $numeric_min !== NULL && $multiplied_data < $numeric_min;
    $is_greater_than_max = !$ignore_max && $numeric_max !== NULL && $multiplied_data > $numeric_max;

    if ($is_less_than_min || $is_greater_than_max) {
      return $this->getFallbackValue($return_type, $default_value, $fallback_source, $item, $numeric_multiply);
    }

    return $data;
  }

  /**
   * Get the fallback value based on return type.
   *
   * @param string $return_type
   *   The return type: 'default_value' or 'other_source'.
   * @param mixed $default_value
   *   The default value.
   * @param string $fallback_source
   *   The fallback source machine name.
   * @param \Drupal\tamper\TamperableItemInterface|null $item
   *   The tamperable item.
   * @param float $multiply
   *   The multiplication factor to apply.
   *
   * @return mixed
   *   The fallback value (multiplied if numeric).
   */
  protected function getFallbackValue(string $return_type, $default_value, string $fallback_source, ?TamperableItemInterface $item = NULL, float $multiply = 1): mixed
  {
    $fallback_value = $default_value;

    if ($return_type === 'other_source' && !empty($fallback_source) && $item) {
      // Try to get the value from the other source
      try {
        $fallback_value = $item->getSource()[$fallback_source] ?? $default_value;
      } catch (\Exception $e) {
        // If we can't get the source value, use default
        $fallback_value = $default_value;
      }
    }

    // Apply multiplication to the fallback value if it's numeric
    if (is_numeric($fallback_value)) {
      $fallback_value = (float) $fallback_value * $multiply;
    }

    return $fallback_value;
  }

  /**
   * Builds a list of selectable source options from the feed mapping.
   *
   * @return array
   *   An associative array of source machine names to labels.
   */
  protected function getSourceOptions(): array
  {
    if (!$this->sourceDefinition) {
      return [];
    }

    $options = $this->sourceDefinition->getList();
    if (!is_array($options) || $options === []) {
      return [];
    }

    $keys = array_keys($options);
    $is_associative = $keys !== range(0, count($keys) - 1);

    if (!$is_associative) {
      // When the list is sequential ensure the option value matches the label so
      // selecting from the dropdown stores the expected source machine name.
      $normalized = [];
      foreach ($options as $value) {
        if (!is_scalar($value)) {
          continue;
        }
        $value = trim((string) $value);
        if ($value === '') {
          continue;
        }
        $normalized[$value] = $value;
      }
      return $normalized;
    }

    $normalized = [];
    foreach ($options as $key => $label) {
      if (!is_scalar($key)) {
        continue;
      }
      $key_string = trim((string) $key);
      if ($key_string === '') {
        continue;
      }
      $normalized[$key_string] = is_scalar($label) ? (string) $label : $key_string;
    }

    return $normalized;
  }
}
