<?php

/**
 * @file
 * Token definitions for Supplier Products AI Rewrite module.
 */

use Drupal\Core\Render\BubbleableMetadata;

/**
 * Implements hook_token_info().
 */
function supplier_products_ai_rewrite_token_info()
{
    $types['ai_enhancement'] = [
        'name' => t('AI Enhancement'),
        'description' => t('Tokens related to AI content enhancement.'),
    ];

    $tokens['ai_enhancement']['category_list'] = [
        'name' => t('Category List'),
        'description' => t('A formatted list of available product categories.'),
    ];

    $tokens['ai_enhancement']['brand_list'] = [
        'name' => t('Brand List'),
        'description' => t('A formatted list of available product brands.'),
    ];

    return [
        'types' => $types,
        'tokens' => $tokens,
    ];
}

/**
 * Implements hook_tokens().
 */
function supplier_products_ai_rewrite_tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata)
{
    $replacements = [];

    if ($type === 'ai_enhancement') {
        foreach ($tokens as $name => $original) {
            switch ($name) {
                case 'category_list':
                    $replacements[$original] = _supplier_products_ai_rewrite_get_category_list();
                    break;

                case 'brand_list':
                    $replacements[$original] = _supplier_products_ai_rewrite_get_brand_list();
                    break;
            }
        }
    }

    return $replacements;
}

/**
 * Gets a formatted list of product categories.
 *
 * @return string
 *   A newline-separated list of category names, or empty string if vocabulary
 *   doesn't exist or is empty.
 */
function _supplier_products_ai_rewrite_get_category_list(): string
{
    try {
        $term_storage = \Drupal::entityTypeManager()->getStorage('taxonomy_term');

        // Check if vocabulary exists first.
        $vocabulary_storage = \Drupal::entityTypeManager()->getStorage('taxonomy_vocabulary');
        if (!$vocabulary_storage->load('product_category')) {
            return '';
        }

        // Use FALSE to get term objects with depth property intact.
        $terms = $term_storage->loadTree('product_category', 0, NULL, FALSE);

        $names = [];
        foreach ($terms as $term) {
            $indent = $term->depth > 0 ? str_repeat('  ', $term->depth) : '';
            $names[] = $indent . $term->name;
        }

        return implode("\n", $names);
    } catch (\Exception $e) {
        \Drupal::logger('supplier_products_ai_rewrite')->warning('Failed to load category list: @message', [
            '@message' => $e->getMessage(),
        ]);
        return '';
    }
}

/**
 * Gets a formatted list of product brands.
 *
 * @return string
 *   A newline-separated list of brand names, or empty string if vocabulary
 *   doesn't exist or is empty.
 */
function _supplier_products_ai_rewrite_get_brand_list(): string
{
    try {
        $term_storage = \Drupal::entityTypeManager()->getStorage('taxonomy_term');

        // Check if vocabulary exists first.
        $vocabulary_storage = \Drupal::entityTypeManager()->getStorage('taxonomy_vocabulary');
        if (!$vocabulary_storage->load('product_brand')) {
            return '';
        }

        $terms = $term_storage->loadTree('product_brand', 0, NULL, TRUE);

        $names = [];
        foreach ($terms as $term) {
            $names[] = $term->label();
        }

        return implode("\n", $names);
    } catch (\Exception $e) {
        \Drupal::logger('supplier_products_ai_rewrite')->warning('Failed to load brand list: @message', [
            '@message' => $e->getMessage(),
        ]);
        return '';
    }
}
