<?php

namespace Drupal\supplier_products_ai_rewrite\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\TermStorageInterface;

/**
 * Service for processing AI responses.
 *
 * Handles parsing AI responses, formatting fields, and resolving taxonomy terms.
 */
class ResultProcessor
{

    public const CATEGORY_VOCABULARY = 'product_category';
    public const BRAND_VOCABULARY = 'product_brand';

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The logger factory.
     *
     * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
     */
    protected LoggerChannelFactoryInterface $loggerFactory;

    /**
     * Constructs a new ResultProcessor object.
     */
    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        ConfigFactoryInterface $config_factory,
        LoggerChannelFactoryInterface $logger_factory
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->configFactory = $config_factory;
        $this->loggerFactory = $logger_factory;
    }

    /**
     * Gets the module config.
     */
    protected function getConfig(): ImmutableConfig
    {
        return $this->configFactory->get('supplier_products_ai_rewrite.settings');
    }

    /**
     * Parses the AI response JSON into field values (single item).
     *
     * @param string $raw_response
     *   The raw AI response string.
     * @param array $tasks
     *   The tasks that were requested.
     *
     * @return array
     *   Parsed results with keys like 'title', 'description', etc.
     */
    public function parseAiResponse(string $raw_response, array $tasks): array
    {
        $config = $this->getConfig();
        $results = [];
        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');

        // Clean up the response - remove markdown code blocks if present.
        $json_str = $this->cleanJsonResponse($raw_response);
        if ($json_str === '') {
            return $results;
        }

        $data = json_decode($json_str, TRUE);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $logger->error('Failed to parse AI response as JSON: @error. Response: @response', [
                '@error' => json_last_error_msg(),
                '@response' => substr($raw_response, 0, 500),
            ]);
            return $results;
        }

        return $this->extractFieldValues($data, $tasks, $config);
    }

    /**
     * Parses the batch response into per-product results.
     *
     * @param string $raw_response
     *   The raw AI response.
     * @param array $batchItems
     *   Array of batch items containing 'node', 'tasks'.
     *
     * @return array
     *   Array keyed by node ID (nid) with parsed results.
     */
    public function parseBatchResponse(string $raw_response, array $batchItems): array
    {
        $config = $this->getConfig();
        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');
        $results = [];

        // Clean up the response.
        $json_str = $this->cleanJsonResponse($raw_response);
        if ($json_str === '') {
            return $results;
        }

        $data = json_decode($json_str, TRUE);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $logger->error('Failed to parse batch AI response as JSON: @error', [
                '@error' => json_last_error_msg(),
            ]);
            return $results;
        }

        // Extract products array.
        $products = $data['products'] ?? $data;
        if (!is_array($products)) {
            $logger->error('Batch response missing products array.');
            return $results;
        }

        // Build nid-to-item lookup.
        $nidToItem = [];
        foreach ($batchItems as $item) {
            $nidToItem[$item['node']->id()] = $item;
        }

        foreach ($products as $product) {
            $nid = $product['nid'] ?? NULL;
            if ($nid === NULL || !isset($nidToItem[$nid])) {
                if ((bool) $config->get('debug_enabled')) {
                    $logger->warning('Batch response contains unknown nid: @nid', ['@nid' => $nid]);
                }
                continue;
            }

            $fields = $product['fields'] ?? $product;
            $tasks = $nidToItem[$nid]['tasks'];
            $results[$nid] = $this->extractFieldValuesFromBatch($fields, $tasks, $config);
        }

        return $results;
    }

    /**
     * Cleans JSON response by removing markdown code blocks.
     *
     * @param string $raw_response
     *   The raw response string.
     *
     * @return string
     *   Cleaned JSON string.
     */
    protected function cleanJsonResponse(string $raw_response): string
    {
        $json_str = trim($raw_response);
        if (preg_match('/```(?:json)?\s*([\s\S]*?)\s*```/', $json_str, $matches)) {
            $json_str = $matches[1];
        }
        return $json_str;
    }

    /**
     * Extracts field values from parsed JSON data (single item).
     */
    protected function extractFieldValues(array $data, array $tasks, ImmutableConfig $config): array
    {
        $results = [];

        if (in_array('title', $tasks, TRUE) && $config->get('title_enabled')) {
            $field = $config->get('title_target_field') ?? 'field_ai_title';
            if (!empty($data[$field])) {
                $results['title'] = (string) $data[$field];
            }
        }

        if (in_array('description', $tasks, TRUE) && $config->get('description_enabled')) {
            $field = $config->get('description_target_field') ?? 'field_ai_description';
            if (!empty($data[$field])) {
                $results['description'] = $this->formatRichText((string) $data[$field]);
            }
        }

        if (in_array('attributes', $tasks, TRUE) && $config->get('attributes_enabled')) {
            $field = $config->get('attributes_target_field') ?? 'field_ai_attributes';
            if (!empty($data[$field]) && is_array($data[$field])) {
                $results['attributes'] = $this->formatAttributes($data[$field]);
            }
        }

        if (in_array('categorize', $tasks, TRUE) && $config->get('categories_enabled')) {
            $field = $config->get('categories_target_field') ?? 'field_ai_suggested_categories';
            if (!empty($data[$field]) && is_array($data[$field])) {
                $results['categories'] = $this->resolveCategories($data[$field]);
            }
        }

        if (in_array('brand', $tasks, TRUE) && $config->get('brand_enabled')) {
            $field = $config->get('brand_target_field') ?? 'field_ai_suggested_brand';
            if (!empty($data[$field]) && is_array($data[$field])) {
                $results['brand'] = $this->resolveBrands($data[$field]);
            }
        }

        return $results;
    }

    /**
     * Extracts field values from batch response fields.
     */
    protected function extractFieldValuesFromBatch(array $fields, array $tasks, ImmutableConfig $config): array
    {
        $parsed = [];

        if (in_array('title', $tasks, TRUE)) {
            $titleField = $config->get('title_target_field') ?? 'field_ai_title';
            if (!empty($fields[$titleField])) {
                $parsed['title'] = (string) $fields[$titleField];
            }
        }

        if (in_array('description', $tasks, TRUE)) {
            $descField = $config->get('description_target_field') ?? 'field_ai_description';
            if (!empty($fields[$descField])) {
                $parsed['description'] = $this->formatRichText((string) $fields[$descField]);
            }
        }

        if (in_array('attributes', $tasks, TRUE)) {
            $attrField = $config->get('attributes_target_field') ?? 'field_ai_attributes';
            if (!empty($fields[$attrField]) && is_array($fields[$attrField])) {
                $parsed['attributes'] = $this->formatAttributes($fields[$attrField]);
            }
        }

        if (in_array('categorize', $tasks, TRUE)) {
            $catField = $config->get('categories_target_field') ?? 'field_ai_suggested_categories';
            if (!empty($fields[$catField]) && is_array($fields[$catField])) {
                $parsed['categories'] = $this->resolveCategories($fields[$catField]);
            }
        }

        if (in_array('brand', $tasks, TRUE)) {
            $brandField = $config->get('brand_target_field') ?? 'field_ai_suggested_brand';
            if (!empty($fields[$brandField]) && is_array($fields[$brandField])) {
                $parsed['brand'] = $this->resolveBrands($fields[$brandField]);
            }
        }

        return $parsed;
    }

    /**
     * Applies results to a node using config target fields.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The node to apply results to.
     * @param array $results
     *   The results to apply.
     */
    public function applyResults(NodeInterface $node, array $results): void
    {
        $config = $this->getConfig();

        if (!empty($results['title'])) {
            $field = $config->get('title_target_field') ?: 'field_ai_title';
            if ($node->hasField($field)) {
                $node->set($field, $results['title']);
            }
        }
        if (!empty($results['description'])) {
            $field = $config->get('description_target_field') ?: 'field_ai_description';
            if ($node->hasField($field)) {
                $node->set($field, $results['description']);
            }
        }
        if (!empty($results['attributes'])) {
            $field = $config->get('attributes_target_field') ?: 'field_ai_attributes';
            if ($node->hasField($field)) {
                $node->set($field, $results['attributes']);
            }
        }
        if (!empty($results['categories'])) {
            $field = $config->get('categories_target_field') ?: 'field_ai_suggested_categories';
            if ($node->hasField($field)) {
                $node->set($field, $results['categories']);
            }
        }
        if (!empty($results['brand'])) {
            $field = $config->get('brand_target_field') ?: 'field_ai_suggested_brand';
            if ($node->hasField($field)) {
                $node->set($field, $results['brand']);
            }
        }
    }

    /**
     * Formats rich text description.
     */
    public function formatRichText(string $text): string
    {
        return trim($text);
    }

    /**
     * Formats attributes array for key_value_field.
     */
    public function formatAttributes(array $attributes): array
    {
        $formatted = [];
        foreach ($attributes as $attr) {
            if (is_array($attr) && !empty($attr['name']) && isset($attr['value'])) {
                $formatted[] = [
                    'key' => (string) $attr['name'],
                    'value' => (string) $attr['value'],
                ];
            } elseif (is_string($attr) && strpos($attr, '|') !== FALSE) {
                [$key, $value] = explode('|', $attr, 2);
                $formatted[] = [
                    'key' => trim($key),
                    'value' => trim($value),
                ];
            }
        }
        return $formatted;
    }

    /**
     * Resolves an array of category names to taxonomy term references.
     */
    public function resolveCategories(array $categories): array
    {
        $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
        if (!$term_storage instanceof TermStorageInterface) {
            return [];
        }

        $results = [];
        foreach ($categories as $category_name) {
            $term_name = is_string($category_name) ? trim($category_name) : '';
            if ($term_name === '') {
                continue;
            }
            $terms = $term_storage->loadByProperties([
                'name' => $term_name,
                'vid' => self::CATEGORY_VOCABULARY,
            ]);
            if ($term = reset($terms)) {
                $results[] = ['target_id' => $term->id()];
            }
        }
        return $results;
    }

    /**
     * Resolves an array of brand names to taxonomy term references.
     */
    public function resolveBrands(array $brands): array
    {
        $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
        if (!$term_storage instanceof TermStorageInterface) {
            return [];
        }

        $results = [];
        foreach ($brands as $brand_name) {
            $term_name = is_string($brand_name) ? trim($brand_name) : '';
            if ($term_name === '') {
                continue;
            }
            $terms = $term_storage->loadByProperties([
                'name' => $term_name,
                'vid' => self::BRAND_VOCABULARY,
            ]);
            if ($term = reset($terms)) {
                $results[] = ['target_id' => $term->id()];
            }
        }
        return $results;
    }

    /**
     * Filters tasks based on global settings.
     *
     * @param array $tasks
     *   The requested tasks.
     *
     * @return array
     *   The filtered tasks that are enabled in config.
     */
    public function filterTasksByConfig(array $tasks): array
    {
        $config = $this->getConfig();

        $taskConfigMap = [
            'title' => 'title_enabled',
            'description' => 'description_enabled',
            'attributes' => 'attributes_enabled',
            'categorize' => 'categories_enabled',
            'brand' => 'brand_enabled',
        ];

        $enabledTasks = [];
        foreach (array_unique($tasks) as $task) {
            $configKey = $taskConfigMap[$task] ?? NULL;
            if ($configKey === NULL || $config->get($configKey)) {
                $enabledTasks[] = $task;
            }
        }

        return array_values($enabledTasks);
    }

    /**
     * Gets the result key for a task.
     *
     * Maps task names to their result keys used in parsed output.
     *
     * @param string $task
     *   The task name (title, description, attributes, categorize, brand).
     *
     * @return string
     *   The result key for the task.
     */
    public function getTaskResultKey(string $task): string
    {
        $mapping = [
            'title' => 'title',
            'description' => 'description',
            'attributes' => 'attributes',
            'categorize' => 'categories',
            'brand' => 'brand',
        ];
        return $mapping[$task] ?? $task;
    }
}
