<?php

namespace Drupal\supplier_products_ai_rewrite\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Utility\Token;
use Drupal\node\NodeInterface;

/**
 * Service for building AI prompts.
 *
 * Handles building system/user messages for both single-item and batch processing.
 */
class PromptBuilder
{

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The token service.
     *
     * @var \Drupal\Core\Utility\Token
     */
    protected Token $token;

    /**
     * Constructs a new PromptBuilder object.
     *
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Core\Utility\Token $token
     *   The token service.
     */
    public function __construct(ConfigFactoryInterface $config_factory, Token $token)
    {
        $this->configFactory = $config_factory;
        $this->token = $token;
    }

    /**
     * Gets the module config.
     *
     * @return \Drupal\Core\Config\ImmutableConfig
     *   The module configuration.
     */
    protected function getConfig(): ImmutableConfig
    {
        return $this->configFactory->get('supplier_products_ai_rewrite.settings');
    }

    /**
     * Builds unified prompt messages for all enabled tasks (single item).
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     * @param array $tasks
     *   The tasks to perform.
     *
     * @return array
     *   Array with 'system' and 'user' keys.
     */
    public function buildUnifiedPromptMessages(NodeInterface $node, array $tasks): array
    {
        $config = $this->getConfig();

        // System message: Role, task, context, field instructions, output schema.
        $system_parts = [];

        $role = $config->get('role_description') ?? '';
        if (!empty($role)) {
            $system_parts[] = $role;
        }

        $task_desc = $config->get('task_description') ?? '';
        if (!empty($task_desc)) {
            $system_parts[] = $task_desc;
        }

        $context = $config->get('context_instructions') ?? '';
        if (!empty($context)) {
            $system_parts[] = $context;
        }

        // Add field-specific instructions for each enabled task.
        $field_instructions = $this->buildFieldInstructions($tasks, $config);
        if (!empty($field_instructions)) {
            $system_parts[] = implode("\n\n", $field_instructions);
        }

        // Add site data (categories, brands) - same for all products.
        $site_data = $this->buildSiteData();
        if (!empty($site_data)) {
            $system_parts[] = $site_data;
        }

        // Add JSON output schema if format is set to 'json'.
        $prompt_format = $config->get('prompt_format') ?? 'json';
        if ($prompt_format === 'json') {
            // Build output schema based on enabled tasks.
            $schema = $this->buildOutputSchema($tasks, $config);
            $system_parts[] = 'Yanıtını aşağıdaki JSON formatında ver. Başka bir şey yazma, sadece JSON döndür:';
            $system_parts[] = json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }

        // User message: Only product-specific input data.
        $user_content = $this->buildSupplierData($node);

        return [
            'system' => implode("\n\n", $system_parts),
            'user' => $user_content,
        ];
    }

    /**
     * Builds a batch prompt for multiple products.
     *
     * @param array $batchItems
     *   Array of batch items, each containing 'node', 'tasks', 'supplier'.
     *
     * @return array
     *   Messages array with 'system' and 'user' keys.
     */
    public function buildBatchPrompt(array $batchItems): array
    {
        $config = $this->getConfig();

        // System message: Role, task, context, field instructions.
        $system_parts = [];

        $role = $config->get('role_description') ?? '';
        if (!empty($role)) {
            $system_parts[] = $role;
        }

        $task_desc = $config->get('task_description') ?? '';
        if (!empty($task_desc)) {
            $system_parts[] = $task_desc;
        }

        $context = $config->get('context_instructions') ?? '';
        if (!empty($context)) {
            $system_parts[] = $context;
        }

        // Collect all unique tasks across all items.
        $allTasks = [];
        foreach ($batchItems as $item) {
            $allTasks = array_merge($allTasks, $item['tasks']);
        }
        $allTasks = array_unique($allTasks);

        // Add field-specific instructions.
        $field_instructions = $this->buildFieldInstructions($allTasks, $config);
        if (!empty($field_instructions)) {
            $system_parts[] = implode("\n\n", $field_instructions);
        }

        // Add site data (categories, brands) - same for all products.
        $site_data = $this->buildSiteData();
        if (!empty($site_data)) {
            $system_parts[] = $site_data;
        }

        // Add JSON output schema if format is set to 'json'.
        $prompt_format = $config->get('prompt_format') ?? 'json';
        if ($prompt_format === 'json') {
            // Build output schema for batch (array of products keyed by node ID).
            $singleSchema = $this->buildOutputSchema($allTasks, $config);
            $exampleNid = reset($batchItems)['node']->id();
            $batchSchema = [
                'products' => [
                    [
                        'nid' => (int) $exampleNid,
                        'fields' => $singleSchema,
                    ],
                ],
            ];

            $system_parts[] = 'Birden fazla ürün için işlem yapıyorsun. Her ürün için verilen node ID (nid) numarasını kullanarak yanıt ver.';
            $system_parts[] = 'Yanıtını aşağıdaki JSON formatında ver. Başka bir şey yazma, sadece JSON döndür:';
            $system_parts[] = json_encode($batchSchema, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        } else {
            $system_parts[] = 'Birden fazla ürün için işlem yapıyorsun. Her ürün için verilen node ID (nid) numarasını kullanarak yanıt ver.';
        }

        // User message: Only product-specific data for each item.
        $user_parts = [];
        foreach ($batchItems as $item) {
            $node = $item['node'];
            $nid = $node->id();
            $user_parts[] = "--- Ürün (nid: $nid) ---";
            $user_parts[] = $this->buildSupplierData($node);
        }

        return [
            'system' => implode("\n\n", $system_parts),
            'user' => implode("\n\n", $user_parts),
        ];
    }

    /**
     * Builds the supplier-specific data string for a product.
     *
     * This contains product-specific information that varies per item.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return string
     *   The supplier data string with tokens replaced.
     */
    public function buildSupplierData(NodeInterface $node): string
    {
        $config = $this->getConfig();
        $supplier_data_template = $config->get('supplier_data') ?? '';

        // Replace tokens with actual node values.
        $data = ['node' => $node];
        $options = ['clear' => TRUE];

        return trim($this->token->replace($supplier_data_template, $data, $options));
    }

    /**
     * Builds the site-wide data string.
     *
     * This contains information that is the same for all products
     * (categories, brands, etc.) and should be included in the system message.
     *
     * @return string
     *   The site data string with tokens replaced.
     */
    public function buildSiteData(): string
    {
        $config = $this->getConfig();
        $site_data_template = $config->get('site_data') ?? '';

        // Site data tokens don't need node context.
        $options = ['clear' => TRUE];

        return trim($this->token->replace($site_data_template, [], $options));
    }

    /**
     * Builds the input data string by merging supplier_data and site_data.
     *
     * @deprecated Use buildSupplierData() and buildSiteData() separately.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return string
     *   The merged input data string.
     */
    public function buildInputData(NodeInterface $node): string
    {
        return $this->buildSupplierData($node) . "\n\n" . $this->buildSiteData();
    }

    /**
     * Builds field-specific instructions for each enabled task.
     *
     * @param array $tasks
     *   The tasks to build instructions for.
     * @param \Drupal\Core\Config\ImmutableConfig $config
     *   The module configuration.
     *
     * @return array
     *   Array of instruction strings.
     */
    protected function buildFieldInstructions(array $tasks, ImmutableConfig $config): array
    {
        $field_instructions = [];

        if (in_array('title', $tasks, TRUE) && $config->get('title_enabled')) {
            $field_instructions[] = 'Ürün Başlığı için: ' . $config->get('rewrite_title_prompt');
        }

        if (in_array('description', $tasks, TRUE) && $config->get('description_enabled')) {
            $field_instructions[] = 'Ürün Açıklaması için: ' . $config->get('rewrite_description_prompt');
        }

        if (in_array('attributes', $tasks, TRUE) && $config->get('attributes_enabled')) {
            $field_instructions[] = 'Özellikler için: ' . $config->get('extract_attributes_prompt');
        }

        if (in_array('categorize', $tasks, TRUE) && $config->get('categories_enabled')) {
            $field_instructions[] = 'Kategoriler için: ' . $config->get('suggest_categories_prompt');
        }

        if (in_array('brand', $tasks, TRUE) && $config->get('brand_enabled')) {
            $field_instructions[] = 'Marka için: ' . $config->get('suggest_brand_prompt');
        }

        return $field_instructions;
    }

    /**
     * Builds the output schema based on enabled tasks.
     *
     * @param array $tasks
     *   The tasks to include in schema.
     * @param \Drupal\Core\Config\ImmutableConfig $config
     *   The module configuration.
     *
     * @return array
     *   The JSON schema structure.
     */
    public function buildOutputSchema(array $tasks, ImmutableConfig $config): array
    {
        $schema = [];

        if (in_array('title', $tasks, TRUE) && $config->get('title_enabled')) {
            $field = $config->get('title_target_field') ?? 'field_ai_title';
            $schema[$field] = 'String - Yeniden yazılmış ürün başlığı';
        }

        if (in_array('description', $tasks, TRUE) && $config->get('description_enabled')) {
            $field = $config->get('description_target_field') ?? 'field_ai_description';
            $schema[$field] = 'HTML formatında ürün açıklaması';
        }

        if (in_array('attributes', $tasks, TRUE) && $config->get('attributes_enabled')) {
            $field = $config->get('attributes_target_field') ?? 'field_ai_attributes';
            $schema[$field] = [['name' => 'Özellik adı', 'value' => 'Özellik değeri']];
        }

        if (in_array('categorize', $tasks, TRUE) && $config->get('categories_enabled')) {
            $field = $config->get('categories_target_field') ?? 'field_ai_suggested_categories';
            $schema[$field] = ['Kategori 1', 'Kategori 2'];
        }

        if (in_array('brand', $tasks, TRUE) && $config->get('brand_enabled')) {
            $field = $config->get('brand_target_field') ?? 'field_ai_suggested_brand';
            $schema[$field] = ['Marka adı'];
        }

        return $schema;
    }
}
