<?php

namespace Drupal\supplier_products_ai_rewrite\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Utility\Token;
use Drupal\node\NodeInterface;

/**
 * Service for managing AI result caching.
 *
 * Handles cache key generation, reading, and writing with atomic operations.
 */
class CacheManager
{

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The file system service.
     *
     * @var \Drupal\Core\File\FileSystemInterface
     */
    protected FileSystemInterface $fileSystem;

    /**
     * The token service.
     *
     * @var \Drupal\Core\Utility\Token
     */
    protected Token $token;

    /**
     * The logger factory.
     *
     * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
     */
    protected LoggerChannelFactoryInterface $loggerFactory;

    /**
     * Static cache for context hash.
     *
     * @var string|null
     */
    protected static ?string $contextHashCache = NULL;

    /**
     * Constructs a new CacheManager object.
     */
    public function __construct(
        ConfigFactoryInterface $config_factory,
        FileSystemInterface $file_system,
        Token $token,
        LoggerChannelFactoryInterface $logger_factory
    ) {
        $this->configFactory = $config_factory;
        $this->fileSystem = $file_system;
        $this->token = $token;
        $this->loggerFactory = $logger_factory;
    }

    /**
     * Gets the module config.
     */
    protected function getConfig(): ImmutableConfig
    {
        return $this->configFactory->get('supplier_products_ai_rewrite.settings');
    }

    /**
     * Gets the context hash (role + task + context_instructions).
     *
     * @return string
     *   Truncated MD5 hash (16 chars).
     */
    public function getContextHash(): string
    {
        if (self::$contextHashCache !== NULL) {
            return self::$contextHashCache;
        }

        $config = $this->getConfig();
        $context = ($config->get('role_description') ?? '') .
            ($config->get('task_description') ?? '') .
            ($config->get('context_instructions') ?? '');

        self::$contextHashCache = substr(md5($context), 0, 16);
        return self::$contextHashCache;
    }

    /**
     * Gets the field prompt hash based on global config.
     *
     * @param string $field
     *   The field type (title, description, attributes, categorize, brand).
     *
     * @return string
     *   Truncated MD5 hash (16 chars).
     */
    public function getFieldPromptHash(string $field): string
    {
        $config = $this->getConfig();

        $prompt_keys = [
            'title' => 'rewrite_title_prompt',
            'description' => 'rewrite_description_prompt',
            'attributes' => 'extract_attributes_prompt',
            'categorize' => 'suggest_categories_prompt',
            'brand' => 'suggest_brand_prompt',
        ];

        $prompt = $config->get($prompt_keys[$field] ?? '') ?? '';

        return substr(md5($prompt), 0, 16);
    }

    /**
     * Gets the supplier data hash (supplier_data tokens replaced).
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return string
     *   Truncated MD5 hash (16 chars).
     */
    public function getSupplierDataHash(NodeInterface $node): string
    {
        $config = $this->getConfig();
        $supplier_data_template = $config->get('supplier_data') ?? '';

        $data = ['node' => $node];
        $options = ['clear' => TRUE];
        $supplier_data = $this->token->replace($supplier_data_template, $data, $options);

        return substr(md5($supplier_data), 0, 16);
    }

    /**
     * Builds the cache file path for a specific field.
     *
     * @param string $contextHash
     *   The context hash.
     * @param string $field
     *   The field name (title, description, etc.).
     * @param string $fieldPromptHash
     *   The field prompt hash.
     * @param string $supplierDataHash
     *   The supplier data hash.
     *
     * @return string
     *   The full cache file path.
     */
    public function getCachePath(string $contextHash, string $field, string $fieldPromptHash, string $supplierDataHash): string
    {
        $config = $this->getConfig();
        $storage = $config->get('cache_storage') ?? 'public';
        $cacheDir = $storage . '://aicache';
        return $cacheDir . '/' . $contextHash . '/' . $field . '_' . $fieldPromptHash . '/' . $supplierDataHash . '.json';
    }

    /**
     * Gets the task-to-field mapping from config.
     *
     * @return array
     *   Map of task names to target field names.
     */
    public function getFieldMapping(): array
    {
        $config = $this->getConfig();
        return [
            'title' => $config->get('title_target_field') ?? 'field_ai_title',
            'description' => $config->get('description_target_field') ?? 'field_ai_description',
            'attributes' => $config->get('attributes_target_field') ?? 'field_ai_attributes',
            'categorize' => $config->get('categories_target_field') ?? 'field_ai_suggested_categories',
            'brand' => $config->get('brand_target_field') ?? 'field_ai_suggested_brand',
        ];
    }

    /**
     * Loads cached results for the given tasks.
     *
     * @param array $tasks
     *   The tasks to check cache for.
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return array
     *   Array with 'cached' and 'missing' keys.
     */
    public function loadFromCache(array $tasks, NodeInterface $node): array
    {
        $contextHash = $this->getContextHash();
        $supplierDataHash = $this->getSupplierDataHash($node);
        $fieldMapping = $this->getFieldMapping();

        $config = $this->getConfig();
        $debugEnabled = (bool) $config->get('debug_enabled');
        $logger = $debugEnabled ? $this->loggerFactory->get('supplier_products_ai_rewrite') : NULL;

        $cached = [];
        $missing = [];

        foreach ($tasks as $task) {
            $fieldPromptHash = $this->getFieldPromptHash($task);
            $cachePath = $this->getCachePath($contextHash, $task, $fieldPromptHash, $supplierDataHash);
            $realPath = $this->fileSystem->realpath($cachePath);

            if ($realPath && file_exists($realPath)) {
                $content = file_get_contents($realPath);
                $data = json_decode($content, TRUE);
                if ($data && isset($data['value'])) {
                    $targetField = $fieldMapping[$task] ?? '';
                    if ($targetField) {
                        $cached[$targetField] = $data['value'];
                        if ($logger) {
                            $logger->info('Cache HIT for task @task on node @nid', [
                                '@task' => $task,
                                '@nid' => $node->id(),
                            ]);
                        }
                    }
                } else {
                    $missing[] = $task;
                    if ($logger) {
                        $logger->info('Cache MISS (invalid data) for task @task on node @nid', [
                            '@task' => $task,
                            '@nid' => $node->id(),
                        ]);
                    }
                }
            } else {
                $missing[] = $task;
                if ($logger) {
                    $logger->info('Cache MISS (not found) for task @task on node @nid', [
                        '@task' => $task,
                        '@nid' => $node->id(),
                    ]);
                }
            }
        }

        return ['cached' => $cached, 'missing' => $missing];
    }

    /**
     * Saves a field result to cache with atomic write operations.
     *
     * @param string $task
     *   The task name (title, description, etc.).
     * @param mixed $value
     *   The value to cache.
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     */
    public function saveToCache(string $task, $value, NodeInterface $node): void
    {
        $contextHash = $this->getContextHash();
        $fieldPromptHash = $this->getFieldPromptHash($task);
        $supplierDataHash = $this->getSupplierDataHash($node);
        $cachePath = $this->getCachePath($contextHash, $task, $fieldPromptHash, $supplierDataHash);

        // Ensure directory exists.
        $directory = dirname($cachePath);
        $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

        $data = [
            'created' => date('c'),
            'value' => $value,
        ];

        // Use atomic write with exclusive locking to prevent race conditions.
        $jsonData = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        $realPath = $this->fileSystem->realpath($directory) . '/' . basename($cachePath);

        // Write to temp file first, then atomic rename.
        $tempFile = $realPath . '.tmp.' . getmypid();
        if (file_put_contents($tempFile, $jsonData, LOCK_EX) !== FALSE) {
            rename($tempFile, $realPath);
        } else {
            // Fallback to Drupal's method if atomic write fails.
            $this->fileSystem->saveData($jsonData, $cachePath, FileSystemInterface::EXISTS_REPLACE);
        }

        // Log cache save if debug mode is enabled.
        $config = $this->getConfig();
        if ($config->get('debug_enabled')) {
            $this->loggerFactory->get('supplier_products_ai_rewrite')->info('Cache SAVED for task @task on node @nid', [
                '@task' => $task,
                '@nid' => $node->id(),
            ]);
        }
    }

    /**
     * Resets the static context hash cache.
     *
     * Should be called if config changes during same request.
     */
    public function resetContextCache(): void
    {
        self::$contextHashCache = NULL;
    }
}
