<?php

namespace Drupal\supplier_products_ai_rewrite\Plugin\views\filter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filters by AI enhancement status using efficient SQL EXISTS subqueries.
 *
 * @ViewsFilter("ai_enhancement_status")
 */
class AiEnhancementStatusFilter extends FilterPluginBase
{

    /**
     * The database connection.
     *
     * @var \Drupal\Core\Database\Connection
     */
    protected Connection $database;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * Status constants.
     */
    const STATUS_ALL = 'all';
    const STATUS_NOT_ENHANCED = 'not_enhanced';
    const STATUS_PARTIALLY_ENHANCED = 'partially_enhanced';
    const STATUS_FULLY_ENHANCED = 'fully_enhanced';

    /**
     * {@inheritdoc}
     */
    public function __construct(array $configuration, $plugin_id, $plugin_definition, Connection $database, ConfigFactoryInterface $config_factory)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->database = $database;
        $this->configFactory = $config_factory;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('database'),
            $container->get('config.factory')
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function defineOptions(): array
    {
        $options = parent::defineOptions();
        $options['value'] = ['default' => self::STATUS_ALL];
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function adminSummary(): string
    {
        if ($this->value === self::STATUS_ALL) {
            return $this->t('All');
        }
        $options = $this->getStatusOptions();
        return $options[$this->value] ?? $this->value;
    }

    /**
     * {@inheritdoc}
     */
    protected function valueForm(&$form, FormStateInterface $form_state): void
    {
        $form['value'] = [
            '#type' => 'select',
            '#title' => $this->t('AI Enhancement Status'),
            '#options' => $this->getStatusOptions(),
            '#default_value' => $this->value,
        ];
    }

    /**
     * Returns status filter options.
     *
     * Note: Views automatically adds "- Any -" for exposed filters,
     * so we don't include it here to avoid duplication.
     */
    protected function getStatusOptions(): array
    {
        return [
            self::STATUS_NOT_ENHANCED => $this->t('Not Enhanced'),
            self::STATUS_PARTIALLY_ENHANCED => $this->t('Partially Enhanced'),
            self::STATUS_FULLY_ENHANCED => $this->t('Fully Enhanced'),
        ];
    }

    /**
     * Gets enabled AI field configuration for performance.
     *
     * Uses static cache to avoid repeated config loads within same request.
     *
     * @return array
     *   Array of enabled fields with their database table/column info.
     */
    protected function getEnabledFieldsConfig(): array
    {
        static $cache = NULL;
        if ($cache !== NULL) {
            return $cache;
        }

        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');

        // Map config keys to their target field names and types.
        $fieldMapping = [
            'title' => [
                'enabled_key' => 'title_enabled',
                'target_key' => 'title_target_field',
                'default_field' => 'field_ai_title',
                'type' => 'text',
            ],
            'description' => [
                'enabled_key' => 'description_enabled',
                'target_key' => 'description_target_field',
                'default_field' => 'field_ai_description',
                'type' => 'text',
            ],
            'attributes' => [
                'enabled_key' => 'attributes_enabled',
                'target_key' => 'attributes_target_field',
                'default_field' => 'field_ai_attributes',
                'type' => 'key_value',
            ],
            'categories' => [
                'enabled_key' => 'categories_enabled',
                'target_key' => 'categories_target_field',
                'default_field' => 'field_ai_suggested_categories',
                'type' => 'entity_reference',
            ],
            'brand' => [
                'enabled_key' => 'brand_enabled',
                'target_key' => 'brand_target_field',
                'default_field' => 'field_ai_suggested_brand',
                'type' => 'entity_reference',
            ],
        ];

        $enabledFields = [];
        foreach ($fieldMapping as $key => $info) {
            $enabled = $config->get($info['enabled_key']) ?? TRUE;
            if (!$enabled) {
                continue;
            }

            $targetField = $config->get($info['target_key']) ?? $info['default_field'];
            $enabledFields[$key] = [
                'field_name' => $targetField,
                'type' => $info['type'],
                'table' => 'node__' . $targetField,
                'value_column' => $this->getValueColumn($info['type']),
            ];
        }

        $cache = $enabledFields;
        return $cache;
    }

    /**
     * Gets the value column name based on field type.
     */
    protected function getValueColumn(string $type): string
    {
        switch ($type) {
            case 'entity_reference':
                return 'target_id';
            case 'key_value':
                return 'key'; // key_value_field uses 'key' column
            default:
                return 'value';
        }
    }

    /**
     * {@inheritdoc}
     */
    public function acceptExposedInput($input): bool
    {
        $rc = parent::acceptExposedInput($input);

        // Normalize the value - Views may pass it differently.
        if (is_array($this->value)) {
            $this->value = reset($this->value) ?: '';
        }

        return $rc;
    }

    /**
     * {@inheritdoc}
     */
    public function query(): void
    {
        // Get the actual value, handling array format.
        $value = is_array($this->value) ? reset($this->value) : $this->value;

        // Skip if no value or "all" selected.
        if (empty($value) || $value === self::STATUS_ALL || $value === 'All') {
            return;
        }

        $enabledFields = $this->getEnabledFieldsConfig();
        if (empty($enabledFields)) {
            // No AI fields enabled, no filtering possible.
            return;
        }

        // Build efficient EXISTS subqueries for each enabled field.
        // Each subquery checks if the field table has a non-null value for the entity.
        $existsConditions = [];
        foreach ($enabledFields as $key => $fieldInfo) {
            $tableName = $fieldInfo['table'];
            $valueColumn = $fieldInfo['field_name'] . '_' . $fieldInfo['value_column'];

            // Check if table exists (field might not be installed)
            if (!$this->database->schema()->tableExists($tableName)) {
                continue;
            }

            // Build EXISTS subquery using node_field_data as base.
            // Views uses node_field_data as the base table for node views.
            $existsConditions[$key] = "EXISTS (
                SELECT 1 FROM {" . $tableName . "} sub_" . $key . "
                WHERE sub_" . $key . ".entity_id = node_field_data.nid
                AND sub_" . $key . ".deleted = 0
                AND sub_" . $key . "." . $valueColumn . " IS NOT NULL
                AND sub_" . $key . "." . $valueColumn . " <> ''
            )";
        }

        if (empty($existsConditions)) {
            return;
        }

        $where = '';
        switch ($value) {
            case self::STATUS_NOT_ENHANCED:
                // NONE of the AI fields are populated.
                // All EXISTS must be FALSE.
                $conditions = array_map(fn($cond) => "NOT {$cond}", $existsConditions);
                $where = '(' . implode(' AND ', $conditions) . ')';
                break;

            case self::STATUS_PARTIALLY_ENHANCED:
                // SOME (but not all) AI fields are populated.
                // At least one EXISTS is TRUE AND at least one is FALSE.
                $atLeastOne = '(' . implode(' OR ', $existsConditions) . ')';
                $notAllConditions = array_map(fn($cond) => "NOT {$cond}", $existsConditions);
                $notAll = '(' . implode(' OR ', $notAllConditions) . ')';
                $where = "({$atLeastOne} AND {$notAll})";
                break;

            case self::STATUS_FULLY_ENHANCED:
                // ALL enabled AI fields are populated.
                // All EXISTS must be TRUE.
                $where = '(' . implode(' AND ', $existsConditions) . ')';
                break;

            default:
                return;
        }

        if (!empty($where)) {
            $this->query->addWhereExpression($this->options['group'], $where);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getCacheTags(): array
    {
        $tags = parent::getCacheTags();
        // Invalidate cache when module config changes (affects enabled fields).
        $tags[] = 'config:supplier_products_ai_rewrite.settings';
        return $tags;
    }
}
