<?php

namespace Drupal\supplier_products_ai_rewrite\Plugin\views\field;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\node\NodeInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Computes AI enhancement status based on populated AI fields.
 *
 * @ViewsField("ai_enhancement_status")
 */
class AiEnhancementStatus extends FieldPluginBase
{

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * {@inheritdoc}
     */
    public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->configFactory = $config_factory;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('config.factory')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // This is a computed field, no database query needed.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $node = $this->getEntity($values);
        if (!$node instanceof NodeInterface || $node->bundle() !== 'supplier_product') {
            return '';
        }

        $status = $this->computeStatus($node);
        return $this->renderStatus($status);
    }

    /**
     * Computes the AI enhancement status for a node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return array
     *   Array of field statuses with 'key', 'label', 'enabled', and 'populated'.
     */
    protected function computeStatus(NodeInterface $node): array
    {
        // Static cache for field configuration (retrieved from config once).
        static $fieldConfig = NULL;
        if ($fieldConfig === NULL) {
            $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');
            $fieldConfig = [
                'title' => [
                    'label' => $this->t('Title'),
                    'target' => $config->get('title_target_field'),
                    'enabled' => (bool) $config->get('title_enabled'),
                ],
                'description' => [
                    'label' => $this->t('Description'),
                    'target' => $config->get('description_target_field'),
                    'enabled' => (bool) $config->get('description_enabled'),
                ],
                'attributes' => [
                    'label' => $this->t('Attributes'),
                    'target' => $config->get('attributes_target_field'),
                    'enabled' => (bool) $config->get('attributes_enabled'),
                ],
                'categories' => [
                    'label' => $this->t('Categories'),
                    'target' => $config->get('categories_target_field'),
                    'enabled' => (bool) $config->get('categories_enabled'),
                ],
                'brand' => [
                    'label' => $this->t('Brand'),
                    'target' => $config->get('brand_target_field'),
                    'enabled' => (bool) $config->get('brand_enabled'),
                ],
            ];
        }

        $statuses = [];

        foreach ($fieldConfig as $key => $info) {
            $targetField = $info['target'];
            // Skip fields that don't exist on the content type.
            if (!$targetField || !$node->hasField($targetField)) {
                continue;
            }

            $isPopulated = !$node->get($targetField)->isEmpty();
            $isEnabled = $info['enabled'];

            $statuses[] = [
                'key' => $key,
                'label' => $info['label'],
                'enabled' => $isEnabled,
                'populated' => $isPopulated,
            ];
        }

        // Sort: Enabled fields first, then disabled fields.
        usort($statuses, function ($a, $b) {
            return $b['enabled'] <=> $a['enabled'];
        });

        return $statuses;
    }

    /**
     * Renders the status with appropriate styling.
     *
     * @param array $statuses
     *   Array of field statuses.
     *
     * @return array
     *   Render array with styled status.
     */
    protected function renderStatus(array $statuses): array
    {
        if (empty($statuses)) {
            return ['#markup' => '-'];
        }

        $boxes = [];
        foreach ($statuses as $status) {
            $title = $status['label'] . ': ';
            $class = 'ai-status-box';

            if (!$status['enabled']) {
                $class .= ' ai-status-box--disabled';
                $title .= $this->t('Disabled');
            } elseif ($status['populated']) {
                $class .= ' ai-status-box--enhanced';
                $title .= $this->t('Enhanced');
            } else {
                $class .= ' ai-status-box--not-enhanced';
                $title .= $this->t('Not Enhanced');
            }

            $boxes[] = "<span class=\"$class\" title=\"$title\"></span>";
        }

        return [
            '#type' => 'inline_template',
            '#template' => '{{ boxes|raw }}',
            '#context' => [
                'boxes' => implode('', $boxes),
            ],
            '#attached' => [
                'library' => ['supplier_products_ai_rewrite/ai-status'],
            ],
        ];
    }
}
