<?php

namespace Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\StreamedChatMessageIteratorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\node\NodeInterface;
use Drupal\supplier_products_ai_rewrite\Service\CacheManager;
use Drupal\supplier_products_ai_rewrite\Service\PromptBuilder;
use Drupal\supplier_products_ai_rewrite\Service\ResultProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes AI content enhancement tasks.
 *
 * @QueueWorker(
 *   id = "ai_content_enhancer",
 *   title = @Translation("AI Content Enhancer")
 * )
 */
class AIContentEnhancer extends QueueWorkerBase implements ContainerFactoryPluginInterface
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The AI provider manager.
     *
     * @var \Drupal\ai\AiProviderPluginManager
     */
    protected AiProviderPluginManager $aiProviderManager;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The logger factory.
     *
     * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
     */
    protected LoggerChannelFactoryInterface $loggerFactory;

    /**
     * The prompt builder service.
     *
     * @var \Drupal\supplier_products_ai_rewrite\Service\PromptBuilder
     */
    protected PromptBuilder $promptBuilder;

    /**
     * The cache manager service.
     *
     * @var \Drupal\supplier_products_ai_rewrite\Service\CacheManager
     */
    protected CacheManager $cacheManager;

    /**
     * The result processor service.
     *
     * @var \Drupal\supplier_products_ai_rewrite\Service\ResultProcessor
     */
    protected ResultProcessor $resultProcessor;

    /**
     * Static buffer for batch processing.
     *
     * @var array
     */
    protected static array $batchBuffer = [];

    /**
     * Constructs an AIContentEnhancer object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
     *   The entity type manager.
     * @param \Drupal\ai\AiProviderPluginManager $aiProviderManager
     *   The AI provider manager.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
     *   The config factory.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
     *   The logger factory.
     * @param \Drupal\supplier_products_ai_rewrite\Service\PromptBuilder $promptBuilder
     *   The prompt builder service.
     * @param \Drupal\supplier_products_ai_rewrite\Service\CacheManager $cacheManager
     *   The cache manager service.
     * @param \Drupal\supplier_products_ai_rewrite\Service\ResultProcessor $resultProcessor
     *   The result processor service.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entityTypeManager,
        AiProviderPluginManager $aiProviderManager,
        ConfigFactoryInterface $configFactory,
        LoggerChannelFactoryInterface $loggerFactory,
        PromptBuilder $promptBuilder,
        CacheManager $cacheManager,
        ResultProcessor $resultProcessor
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entityTypeManager;
        $this->aiProviderManager = $aiProviderManager;
        $this->configFactory = $configFactory;
        $this->loggerFactory = $loggerFactory;
        $this->promptBuilder = $promptBuilder;
        $this->cacheManager = $cacheManager;
        $this->resultProcessor = $resultProcessor;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager'),
            $container->get('ai.provider'),
            $container->get('config.factory'),
            $container->get('logger.factory'),
            $container->get('supplier_products_ai_rewrite.prompt_builder'),
            $container->get('supplier_products_ai_rewrite.cache_manager'),
            $container->get('supplier_products_ai_rewrite.result_processor')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function processItem($item): void
    {
        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');
        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');

        $nid = $item['nid'] ?? NULL;
        $debugEnabled = (bool) $config->get('debug_enabled');

        if (!$nid) {
            if ($debugEnabled) {
                $logger->warning('Skipping queue item with missing node ID.');
            }
            return;
        }

        $node_storage = $this->entityTypeManager->getStorage('node');
        $node = $node_storage->load($nid);

        if (!$node instanceof NodeInterface || $node->bundle() !== 'supplier_product') {
            if ($debugEnabled) {
                $logger->warning('Queue item skipped: Node @nid not found or invalid bundle.', ['@nid' => $nid]);
            }
            return;
        }

        if ($node->get('field_supplier')->isEmpty()) {
            if ($debugEnabled) {
                $logger->warning('Node @nid skipped: No supplier associated.', ['@nid' => $nid]);
            }
            return;
        }

        $supplier = $node->get('field_supplier')->entity;
        if (!$supplier) {
            return;
        }

        // Filter tasks based on global config settings.
        $tasks = $this->resultProcessor->filterTasksByConfig($item['tasks'] ?? []);
        if (empty($tasks)) {
            return;
        }

        // Get batch size - when 1, process immediately; when >1, buffer items.
        $batchSize = (int) ($config->get('batch_size') ?? 1);

        if ($batchSize <= 1) {
            // Single item processing (batch of 1).
            $this->processSingleItem($node, $tasks, $supplier);
            return;
        }

        // Check if we should overwrite cached responses.
        $overwriteCache = !empty($item['overwrite_cache']);

        // Batch mode: Check cache first to see what's needed (unless overwriting).
        if ($overwriteCache) {
            // Skip cache, treat all tasks as missing.
            $cached = [];
            $missing = $tasks;
            if ($config->get('debug_enabled')) {
                $logger->info('Overwrite cache enabled for node @nid, skipping cache lookup.', ['@nid' => $node->id()]);
            }
        } else {
            $cacheResult = $this->cacheManager->loadFromCache($tasks, $node);
            $cached = $cacheResult['cached'];
            $missing = $cacheResult['missing'];
        }

        // Apply cached results immediately.
        $this->applyCachedResults($node, $cached);

        // If all tasks are cached, save and skip batch.
        if (empty($missing)) {
            if ($config->get('debug_enabled')) {
                $logger->info('All tasks cached for node @nid, skipping batch.', ['@nid' => $node->id()]);
            }
            if (!empty($cached)) {
                $node->save();
            }
            return;
        }

        // Register shutdown handler on first buffered item.
        static $shutdownRegistered = FALSE;
        if (!$shutdownRegistered && empty(self::$batchBuffer)) {
            register_shutdown_function([self::class, 'flushBatch']);
            $shutdownRegistered = TRUE;
        }

        // Add to batch buffer.
        self::$batchBuffer[] = [
            'node' => $node,
            'tasks' => $missing,
            'supplier' => $supplier,
            'cached' => $cached,
        ];

        // Process batch when buffer is full.
        if (count(self::$batchBuffer) >= $batchSize) {
            $this->processBatch();
        }
    }

    /**
     * Processes a single item (non-batch mode).
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     * @param array $tasks
     *   The tasks to perform.
     * @param mixed $supplier
     *   The supplier taxonomy term.
     */
    protected function processSingleItem(NodeInterface $node, array $tasks, $supplier): void
    {
        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');

        try {
            $results = $this->generateAllFields($node, $tasks, $supplier);
            $this->resultProcessor->applyResults($node, $results);

            if (!empty($results)) {
                $node->save();
            }
        } catch (\Exception $e) {
            $logger->error('AI processing failed for node @nid: @message', [
                '@nid' => $node->id(),
                '@message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Processes the accumulated batch buffer.
     */
    protected function processBatch(): void
    {
        if (empty(self::$batchBuffer)) {
            return;
        }

        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');
        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $debugEnabled = (bool) $config->get('debug_enabled');

        if ($debugEnabled) {
            $logger->info('Processing batch of @count items.', ['@count' => count(self::$batchBuffer)]);
        }

        try {
            // Build unified prompt for all items in batch.
            $batchPrompt = $this->promptBuilder->buildBatchPrompt(self::$batchBuffer);

            // Estimate token size and warn if potentially too large.
            $promptSize = strlen(json_encode($batchPrompt));
            if ($promptSize > 50000) { // ~12.5K tokens estimated.
                $logger->warning('Batch prompt is very large (@size KB). Consider reducing batch_size to avoid API limits.', [
                    '@size' => round($promptSize / 1024, 1),
                ]);
            }

            $raw_response = $this->runChatWithMessages($batchPrompt, '');

            // Parse batch response.
            $batchResults = $this->resultProcessor->parseBatchResponse($raw_response, self::$batchBuffer);

            // Apply results and save each node.
            foreach (self::$batchBuffer as $item) {
                $node = $item['node'];
                $nid = $node->id();
                $tasks = $item['tasks'];

                if (isset($batchResults[$nid])) {
                    $results = $batchResults[$nid];

                    // Save to cache.
                    foreach ($tasks as $task) {
                        $taskKey = $this->resultProcessor->getTaskResultKey($task);
                        if (isset($results[$taskKey]) && !empty($results[$taskKey])) {
                            $this->cacheManager->saveToCache($task, $results[$taskKey], $node);
                        }
                    }

                    // Apply results to node.
                    $this->resultProcessor->applyResults($node, $results);
                    $node->save();

                    if ($debugEnabled) {
                        $logger->info('Batch processed node @nid.', ['@nid' => $node->id()]);
                    }
                } else {
                    // No results for this node - process individually as fallback.
                    if ($debugEnabled) {
                        $logger->warning('No results for node @nid in batch response. Processing individually.', ['@nid' => $node->id()]);
                    }
                    try {
                        $this->processSingleItem($node, $tasks, $item['supplier']);
                    } catch (\Exception $e) {
                        $logger->error('Individual processing also failed for node @nid: @message', [
                            '@nid' => $node->id(),
                            '@message' => $e->getMessage(),
                        ]);
                    }
                }
            }
        } catch (\Exception $e) {
            $logger->error('Batch processing failed: @message', ['@message' => $e->getMessage()]);

            // Fallback: Process each item individually.
            foreach (self::$batchBuffer as $item) {
                try {
                    $this->processSingleItem($item['node'], $item['tasks'], $item['supplier']);
                } catch (\Exception $individualError) {
                    $logger->error('Individual fallback failed for node @nid: @message', [
                        '@nid' => $item['node']->id(),
                        '@message' => $individualError->getMessage(),
                    ]);
                }
            }
        }

        // Clear the buffer.
        self::$batchBuffer = [];
    }

    /**
     * Applies cached results to a node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The node to apply cached results to.
     * @param array $cached
     *   The cached field values.
     */
    protected function applyCachedResults(NodeInterface $node, array $cached): void
    {
        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $results = $this->mapCachedToResults($cached, $config);
        $this->resultProcessor->applyResults($node, $results);
    }

    /**
     * Maps cached field names back to result keys.
     *
     * @param array $cached
     *   Cached values keyed by field name.
     * @param \Drupal\Core\Config\ImmutableConfig $config
     *   The module configuration.
     *
     * @return array
     *   Results keyed by task name.
     */
    protected function mapCachedToResults(array $cached, $config): array
    {
        $results = [];
        $field_to_key = [
            ($config->get('title_target_field') ?? 'field_ai_title') => 'title',
            ($config->get('description_target_field') ?? 'field_ai_description') => 'description',
            ($config->get('attributes_target_field') ?? 'field_ai_attributes') => 'attributes',
            ($config->get('categories_target_field') ?? 'field_ai_suggested_categories') => 'categories',
            ($config->get('brand_target_field') ?? 'field_ai_suggested_brand') => 'brand',
        ];

        foreach ($cached as $field => $value) {
            $key = $field_to_key[$field] ?? $field;
            $results[$key] = $value;
        }

        return $results;
    }

    /**
     * Flushes any remaining items in the batch buffer.
     *
     * Should be called after queue processing is complete.
     */
    public static function flushBatch(): void
    {
        if (empty(self::$batchBuffer)) {
            return;
        }

        try {
            $pluginManager = \Drupal::service('plugin.manager.queue_worker');
            if (!$pluginManager) {
                \Drupal::logger('supplier_products_ai_rewrite')->error('Cannot flush batch: Queue worker manager not available. @count items lost.', [
                    '@count' => count(self::$batchBuffer),
                ]);
                self::$batchBuffer = [];
                return;
            }

            $instance = $pluginManager->createInstance('ai_content_enhancer');
            if ($instance instanceof self) {
                $instance->processBatch();
            } else {
                \Drupal::logger('supplier_products_ai_rewrite')->error('Cannot flush batch: Invalid instance type. @count items lost.', [
                    '@count' => count(self::$batchBuffer),
                ]);
                self::$batchBuffer = [];
            }
        } catch (\Exception $e) {
            \Drupal::logger('supplier_products_ai_rewrite')->error('Batch flush failed: @message. @count items lost.', [
                '@message' => $e->getMessage(),
                '@count' => count(self::$batchBuffer),
            ]);
            self::$batchBuffer = [];
        }
    }

    /**
     * Generates all enabled AI fields, using cache when available.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     * @param array $tasks
     *   The tasks to perform.
     * @param mixed $supplier
     *   The supplier taxonomy term or NULL.
     *
     * @return array
     *   Parsed results with field names as keys.
     */
    protected function generateAllFields(NodeInterface $node, array $tasks, $supplier = NULL): array
    {
        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');

        // Check cache for existing results.
        $cacheResult = $this->cacheManager->loadFromCache($tasks, $node);
        $cached = $cacheResult['cached'];
        $missing = $cacheResult['missing'];

        // Log cache hit info (only in debug mode).
        $debugEnabled = (bool) $config->get('debug_enabled');
        if (!empty($cached) && $debugEnabled) {
            $logger->info('Cache hit for @count tasks on node @nid', [
                '@count' => count($cached),
                '@nid' => $node->id(),
            ]);
        }

        // If all tasks are cached, return cached results.
        if (empty($missing)) {
            if ($debugEnabled) {
                $logger->info('All tasks cached for node @nid, skipping API call.', ['@nid' => $node->id()]);
            }
            return $this->mapCachedToResults($cached, $config);
        }

        // Call API only for missing tasks.
        $messages = $this->promptBuilder->buildUnifiedPromptMessages($node, $missing);
        $original_desc = (string) $node->get('field_supplier_product_descripti')->value;
        $raw_response = $this->runChatWithMessages($messages, $original_desc);

        // Parse the JSON response.
        $apiResults = $this->resultProcessor->parseAiResponse($raw_response, $missing);

        // Save API results to cache.
        foreach ($missing as $task) {
            $taskKey = $this->resultProcessor->getTaskResultKey($task);
            if (isset($apiResults[$taskKey]) && !empty($apiResults[$taskKey])) {
                $this->cacheManager->saveToCache($task, $apiResults[$taskKey], $node);
            }
        }

        // Merge cached results with API results.
        $mergedResults = $this->mapCachedToResults($cached, $config);
        foreach ($apiResults as $key => $value) {
            $mergedResults[$key] = $value;
        }

        return $mergedResults;
    }

    /**
     * Runs a chat prompt with structured system/user messages.
     *
     * @param array $messages
     *   An array with 'system' and 'user' keys containing message content.
     * @param string $original_value
     *   The original value (used for mock responses in debug mode).
     *
     * @return string
     *   The AI response text.
     */
    protected function runChatWithMessages(array $messages, string $original_value = ''): string
    {
        $system_content = trim($messages['system'] ?? '');
        $user_content = trim($messages['user'] ?? '');

        if ($user_content === '') {
            return '';
        }

        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $logger = $this->loggerFactory->get('supplier_products_ai_rewrite');

        // Debug mode: Store prompts for later logging.
        $debugPrompt = '';
        if ($config->get('debug_enabled') && $config->get('debug_log_prompts')) {
            $debugPrompt = json_encode([
                'system' => $system_content,
                'user' => $user_content,
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }

        // Debug mode: Use mock responses instead of calling AI.
        if ($config->get('debug_enabled') && $config->get('debug_use_mock')) {
            $prefix = $config->get('debug_mock_prefix') ?? '[AI-TEST] ';
            $mock_response = $prefix . $original_value;

            if ($config->get('debug_log_responses') || $config->get('debug_log_prompts')) {
                $logger->info('AI Debug:<br><strong>Prompt:</strong><pre>@prompt</pre><br><strong>Response:</strong><pre>@response</pre>', [
                    '@prompt' => $debugPrompt ?: 'N/A',
                    '@response' => $mock_response,
                ]);
            }

            return $mock_response;
        }

        $defaults = $this->aiProviderManager->getDefaultProviderForOperationType('chat');
        if (empty($defaults['provider_id']) || empty($defaults['model_id'])) {
            throw new \RuntimeException('No default AI provider configured for chat operations.');
        }

        $provider = $this->aiProviderManager->createInstance($defaults['provider_id']);
        if (!$provider->isUsable('chat')) {
            throw new \RuntimeException(sprintf('AI provider %s is not ready for chat operations.', $defaults['provider_id']));
        }

        // Build messages based on prompt_format setting.
        $prompt_format = $config->get('prompt_format') ?? 'json';
        $chat_messages = [];

        if ($prompt_format === 'plain_text') {
            // Plain text: Combine system and user content into single user message.
            $combined_content = '';
            if (!empty($system_content)) {
                $combined_content = $system_content . "\n\n---\n\n";
            }
            $combined_content .= $user_content;
            $chat_messages[] = new ChatMessage('user', $combined_content);
        } else {
            // JSON format: Use separate system and user messages.
            if (!empty($system_content)) {
                $chat_messages[] = new ChatMessage('system', $system_content);
            }
            $chat_messages[] = new ChatMessage('user', $user_content);
        }

        $input = new ChatInput($chat_messages);

        $response = $provider->chat($input, $defaults['model_id']);
        $normalized = $response->getNormalized();
        $result = '';

        if ($normalized instanceof ChatMessage) {
            $result = trim($normalized->getText());
        } elseif ($normalized instanceof StreamedChatMessageIteratorInterface) {
            $text = '';
            foreach ($normalized as $chat_message) {
                $text .= $chat_message->getText();
            }
            $result = trim($text);
        }

        // Debug mode: Log combined prompt and response.
        if ($config->get('debug_enabled') && ($config->get('debug_log_prompts') || $config->get('debug_log_responses'))) {
            $logger->info('AI Debug:<br><strong>Prompt:</strong><pre>@prompt</pre><br><strong>Response:</strong><pre>@response</pre>', [
                '@prompt' => $debugPrompt ?: 'N/A',
                '@response' => $result,
            ]);
        }

        return $result;
    }
}
