<?php

namespace Drupal\supplier_products_ai_rewrite\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\node\NodeInterface;
use Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Triggers AI processing for selected Supplier Product nodes.
 *
 * @Action(
 *   id = "supplier_products_ai_rewrite_trigger_ai_enhancement",
 *   label = @Translation("AI Enhance Content"),
 *   type = "node",
 *   confirm = TRUE
 * )
 */
class TriggerAiEnhancementAction extends ViewsBulkOperationsActionBase implements ContainerFactoryPluginInterface
{
    use StringTranslationTrait;

    /**
     * The messenger service.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * The queue worker manager.
     *
     * @var \Drupal\Core\Queue\QueueWorkerManagerInterface
     */
    protected $queueWorkerManager;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a TriggerAiEnhancementAction object.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        MessengerInterface $messenger,
        $queue_worker_manager,
        EntityTypeManagerInterface $entity_type_manager,
        LoggerChannelFactoryInterface $logger_factory
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->messenger = $messenger;
        $this->queueWorkerManager = $queue_worker_manager;
        $this->entityTypeManager = $entity_type_manager;
        $this->logger = $logger_factory->get('supplier_products_ai_rewrite');
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('messenger'),
            $container->get('plugin.manager.queue_worker'),
            $container->get('entity_type.manager'),
            $container->get('logger.factory')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
    {
        // Get global AI settings to determine which features are enabled.
        $config = \Drupal::config('supplier_products_ai_rewrite.settings');

        // Define task options with their corresponding config keys.
        $task_config_map = [
            'title' => 'title_enabled',
            'description' => 'description_enabled',
            'attributes' => 'attributes_enabled',
            'categorize' => 'categories_enabled',
            'brand' => 'brand_enabled',
        ];

        $task_labels = [
            'title' => $this->t('Rewrite Title'),
            'description' => $this->t('Rewrite Description'),
            'attributes' => $this->t('Extract Attributes'),
            'categorize' => $this->t('Suggest Categories'),
            'brand' => $this->t('Suggest Brand'),
        ];

        $options = [];
        $default_value = [];
        $disabled_tasks = [];

        foreach ($task_config_map as $task => $config_key) {
            $is_enabled = $config->get($config_key) ?? TRUE;
            $options[$task] = $task_labels[$task];

            if ($is_enabled) {
                $default_value[] = $task;
            } else {
                $disabled_tasks[] = $task;
            }
        }

        $form['tasks'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('Select content to enhance'),
            '#description' => $this->t('Choose which AI enhancement tasks to run on the selected products.'),
            '#options' => $options,
            '#default_value' => $default_value,
            '#required' => TRUE,
        ];

        // Disable checkboxes for globally disabled features.
        foreach ($disabled_tasks as $task) {
            $form['tasks'][$task] = [
                '#disabled' => TRUE,
                '#default_value' => FALSE,
            ];
        }

        if (!empty($disabled_tasks)) {
            $form['tasks']['#description'] = $this->t('Choose which AI enhancement tasks to run on the selected products. Grayed out options are disabled in the <a href="@settings_url">AI settings</a>.', [
                '@settings_url' => '/admin/commerce/config/synchronization/ai-settings',
            ]);
        }

        $form['clear_existing'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Clear existing AI data before processing'),
            '#description' => $this->t('If checked, existing AI-generated content will be cleared before regenerating. This also removes manual edit locks.'),
            '#default_value' => TRUE,
        ];

        $form['overwrite_cache'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Overwrite cached responses'),
            '#description' => $this->t('If checked, cached AI responses will be ignored and new responses will be generated. Useful when prompts have changed.'),
            '#default_value' => FALSE,
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void
    {
        $tasks = array_filter($form_state->getValue('tasks'));
        if (empty($tasks)) {
            $form_state->setErrorByName('tasks', $this->t('Please select at least one enhancement task.'));
        }
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void
    {
        $this->configuration['tasks'] = array_filter($form_state->getValue('tasks'));
        $this->configuration['clear_existing'] = $form_state->getValue('clear_existing');
        $this->configuration['overwrite_cache'] = $form_state->getValue('overwrite_cache');
    }

    /**
     * {@inheritdoc}
     */
    public function execute($entity = NULL)
    {
        if (!$entity instanceof NodeInterface || $entity->bundle() !== 'supplier_product') {
            return;
        }

        // Get selected tasks from configuration.
        $selected_tasks = $this->configuration['tasks'] ?? [];
        $clear_existing = $this->configuration['clear_existing'] ?? TRUE;
        $overwrite_cache = $this->configuration['overwrite_cache'] ?? FALSE;

        if (empty($selected_tasks)) {
            $this->messenger->addWarning(
                $this->t('No enhancement tasks selected for "@title".', [
                    '@title' => $entity->label(),
                ])
            );
            return;
        }

        // Filter tasks based on available content.
        $available_tasks = [];
        $unavailable_reasons = [];

        if (in_array('title', $selected_tasks, TRUE)) {
            if (!$entity->get('field_original_product_title')->isEmpty()) {
                $available_tasks[] = 'title';
            } else {
                $unavailable_reasons[] = $this->t('title (no original title)');
            }
        }

        $desc_based_tasks = array_intersect(['description', 'attributes', 'categorize', 'brand'], $selected_tasks);
        if (!empty($desc_based_tasks)) {
            if (!$entity->get('field_supplier_product_descripti')->isEmpty()) {
                $available_tasks = array_merge($available_tasks, $desc_based_tasks);
            } else {
                foreach ($desc_based_tasks as $task) {
                    $unavailable_reasons[] = $this->t('@task (no description)', ['@task' => $task]);
                }
            }
        }

        if (empty($available_tasks)) {
            $this->messenger->addWarning(
                $this->t('Product "@title" has no content for selected tasks: @reasons', [
                    '@title' => $entity->label(),
                    '@reasons' => implode(', ', $unavailable_reasons),
                ])
            );
            return;
        }

        // Clear existing AI data if requested.
        // Get config once for use throughout the method.
        $config = \Drupal::config('supplier_products_ai_rewrite.settings');

        if ($clear_existing) {

            if (in_array('title', $available_tasks, TRUE)) {
                $titleField = $config->get('title_target_field') ?? 'field_ai_title';
                if ($entity->hasField($titleField)) {
                    $entity->set($titleField, NULL);
                }
            }
            if (in_array('description', $available_tasks, TRUE)) {
                $descField = $config->get('description_target_field') ?? 'field_ai_description';
                if ($entity->hasField($descField)) {
                    $entity->set($descField, NULL);
                }
            }
            if (in_array('attributes', $available_tasks, TRUE)) {
                $attrField = $config->get('attributes_target_field') ?? 'field_ai_attributes';
                if ($entity->hasField($attrField)) {
                    $entity->set($attrField, []);
                }
            }
            if (in_array('categorize', $available_tasks, TRUE)) {
                $catField = $config->get('categories_target_field') ?? 'field_ai_suggested_categories';
                if ($entity->hasField($catField)) {
                    $entity->set($catField, []);
                }
            }
            if (in_array('brand', $available_tasks, TRUE)) {
                $brandField = $config->get('brand_target_field') ?? 'field_ai_suggested_brand';
                if ($entity->hasField($brandField)) {
                    $entity->set($brandField, []);
                }
            }

            $entity->save();
        }

        // Get current signatures.
        $title_sig = $entity->get('field_title_signature')->value;
        $desc_sig = $entity->get('field_description_signature')->value;

        // Process immediately using the queue worker.
        $queue_worker = $this->queueWorkerManager->createInstance('ai_content_enhancer');

        $item = [
            'nid' => $entity->id(),
            'tasks' => array_values($available_tasks),
            'title_sig' => $title_sig,
            'desc_sig' => $desc_sig,
            'overwrite_cache' => $overwrite_cache,
        ];

        try {
            $queue_worker->processItem($item);

            // Flush any buffered items to ensure processing completes.
            // This is needed when batch_size > 1 as items are buffered.
            AIContentEnhancer::flushBatch();

            // Reload the node to see the changes.
            $reloaded_node = $this->entityTypeManager->getStorage('node')->load($entity->id());
            if (!$reloaded_node instanceof NodeInterface) {
                $this->messenger->addError(
                    $this->t('Unable to reload product "@title" after enhancement.', [
                        '@title' => $entity->label(),
                    ])
                );
                return;
            }

            // Check if AI fields were populated using config field names.
            $changes = [];

            $titleField = $config->get('title_target_field') ?? 'field_ai_title';
            if (in_array('title', $available_tasks, TRUE) && $reloaded_node->hasField($titleField) && !$reloaded_node->get($titleField)->isEmpty()) {
                $changes[] = 'title';
            }

            $descField = $config->get('description_target_field') ?? 'field_ai_description';
            if (in_array('description', $available_tasks, TRUE) && $reloaded_node->hasField($descField) && !$reloaded_node->get($descField)->isEmpty()) {
                $changes[] = 'description';
            }

            $attrField = $config->get('attributes_target_field') ?? 'field_ai_attributes';
            if (in_array('attributes', $available_tasks, TRUE) && $reloaded_node->hasField($attrField) && !$reloaded_node->get($attrField)->isEmpty()) {
                $changes[] = 'attributes';
            }

            $catField = $config->get('categories_target_field') ?? 'field_ai_suggested_categories';
            if (in_array('categorize', $available_tasks, TRUE) && $reloaded_node->hasField($catField) && !$reloaded_node->get($catField)->isEmpty()) {
                $changes[] = 'categories';
            }

            $brandField = $config->get('brand_target_field') ?? 'field_ai_suggested_brand';
            if (in_array('brand', $available_tasks, TRUE) && $reloaded_node->hasField($brandField) && !$reloaded_node->get($brandField)->isEmpty()) {
                $changes[] = 'brand';
            }

            if (!empty($changes)) {
                $this->messenger->addStatus(
                    $this->t('Product "@title" enhanced: @changes', [
                        '@title' => $entity->label(),
                        '@changes' => implode(', ', $changes),
                    ])
                );
            } else {
                $this->messenger->addWarning(
                    $this->t('Product "@title": no content generated for selected tasks.', [
                        '@title' => $entity->label(),
                    ])
                );
            }
        } catch (\Exception $e) {
            $this->logger->error('AI enhancement failed for node @nid: @message. Trace: @trace', [
                '@nid' => $entity->id(),
                '@message' => $e->getMessage(),
                '@trace' => $e->getTraceAsString(),
            ]);
            $this->messenger->addError(
                $this->t('AI enhancement failed for "@title": @message', [
                    '@title' => $entity->label(),
                    '@message' => $e->getMessage(),
                ])
            );
        }
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        if (!$object instanceof NodeInterface) {
            $result = AccessResult::neutral();
            return $return_as_object ? $result : $result->isAllowed();
        }

        $access = $object->access('update', $account, TRUE)
            ->andIf(AccessResult::allowedIf($object->bundle() === 'supplier_product'))
            ->andIf(AccessResult::allowedIfHasPermission($account, 'trigger ai content enhancement'));

        return $return_as_object ? $access : $access->isAllowed();
    }
}
