<?php

namespace Drupal\supplier_products_ai_rewrite\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Utility\Token;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\supplier_products_ai_rewrite\Service\PromptBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for AI content rewrite settings.
 */
class SettingsForm extends ConfigFormBase
{
    /**
     * The entity field manager.
     *
     * @var \Drupal\Core\Entity\EntityFieldManagerInterface
     */
    protected EntityFieldManagerInterface $entityFieldManager;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The token service.
     *
     * @var \Drupal\Core\Utility\Token
     */
    protected Token $token;

    /**
     * The prompt builder service.
     *
     * @var \Drupal\supplier_products_ai_rewrite\Service\PromptBuilder
     */
    protected PromptBuilder $promptBuilder;

    /**
     * The AI provider manager.
     *
     * @var \Drupal\ai\AiProviderPluginManager
     */
    protected AiProviderPluginManager $aiProviderManager;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container): static
    {
        $instance = parent::create($container);
        $instance->entityFieldManager = $container->get('entity_field.manager');
        $instance->entityTypeManager = $container->get('entity_type.manager');
        $instance->token = $container->get('token');
        $instance->promptBuilder = $container->get('supplier_products_ai_rewrite.prompt_builder');
        $instance->aiProviderManager = $container->get('ai.provider');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames(): array
    {
        return ['supplier_products_ai_rewrite.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId(): string
    {
        return 'supplier_products_ai_rewrite_settings_form';
    }

    /**
     * Gets the list of available text fields from supplier_product content type.
     */
    protected function getSupplierProductTextFields(): array
    {
        $options = ['' => $this->t('- Select a field -')];
        $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', 'supplier_product');

        foreach ($field_definitions as $field_name => $definition) {
            // Only include text-based fields suitable for storing AI output.
            $type = $definition->getType();
            if (in_array($type, ['string', 'string_long', 'text', 'text_long', 'text_with_summary'])) {
                $options[$field_name] = $definition->getLabel() . ' (' . $field_name . ')';
            }
        }

        return $options;
    }

    /**
     * Gets the list of available entity reference fields from supplier_product.
     */
    protected function getSupplierProductReferenceFields(): array
    {
        $options = ['' => $this->t('- Select a field -')];
        $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', 'supplier_product');

        foreach ($field_definitions as $field_name => $definition) {
            $type = $definition->getType();
            if ($type === 'entity_reference') {
                $options[$field_name] = $definition->getLabel() . ' (' . $field_name . ')';
            }
        }

        return $options;
    }

    /**
     * Gets the list of available key-value fields from supplier_product.
     */
    protected function getSupplierProductKeyValueFields(): array
    {
        $options = ['' => $this->t('- Select a field -')];
        $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', 'supplier_product');

        foreach ($field_definitions as $field_name => $definition) {
            $type = $definition->getType();
            // Include key_value fields.
            if (str_contains($type, 'key_value') || $type === 'map') {
                $options[$field_name] = $definition->getLabel() . ' (' . $field_name . ')';
            }
        }

        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state): array
    {
        $config = $this->config('supplier_products_ai_rewrite.settings');
        $text_fields = $this->getSupplierProductTextFields();
        $reference_fields = $this->getSupplierProductReferenceFields();
        $key_value_fields = $this->getSupplierProductKeyValueFields();

        $form['enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable content enhancement with AI'),
            '#description' => $this->t('When disabled, AI content enhancement will not run.'),
            '#default_value' => $config->get('enabled') ?? TRUE,
        ];

        $form['enhance_on_create'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enhance product data with AI when creating commerce product'),
            '#description' => $this->t('Automatically run AI enhancement when a commerce product is created from supplier product data.'),
            '#default_value' => $config->get('enhance_on_create') ?? FALSE,
        ];

        $form['enhance_on_merge'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enhance product data with AI when merging and creating commerce product'),
            '#description' => $this->t('Automatically run AI enhancement when merging data from multiple suppliers and creating a commerce product.'),
            '#default_value' => $config->get('enhance_on_merge') ?? FALSE,
        ];

        $form['batch_size'] = [
            '#type' => 'number',
            '#title' => $this->t('Batch size'),
            '#description' => $this->t('Number of products to process in a single API call. Set to 1 for single item processing, or higher (3-5) to reduce API calls and improve throughput.'),
            '#default_value' => $config->get('batch_size') ?? 1,
            '#min' => 1,
            '#max' => 20,
        ];

        $form['prompt_format'] = [
            '#type' => 'select',
            '#title' => $this->t('Prompt output format'),
            '#description' => $this->t('Select the format for AI prompts. JSON includes a structured schema for consistent output; Plain text omits the schema.'),
            '#options' => [
                'json' => $this->t('JSON (with output schema)'),
                'plain_text' => $this->t('Plain text (without schema)'),
            ],
            '#default_value' => $config->get('prompt_format') ?? 'json',
        ];

        $form['role_description'] = [
            '#type' => 'textarea',
            '#title' => $this->t('AI Role Description'),
            '#description' => $this->t('Define the role or persona of the AI. This will be prepended to all prompts as a system instruction. Example: "You are a professional e-commerce copywriter specializing in product descriptions."'),
            '#default_value' => $config->get('role_description') ?? '',
            '#rows' => 3,
        ];

        $form['task_description'] = [
            '#type' => 'textarea',
            '#title' => $this->t('AI Task Description'),
            '#description' => $this->t('Define the general task or objective for the AI. This will be included in the prompt context.'),
            '#default_value' => $config->get('task_description') ?? '',
            '#rows' => 3,
        ];

        $form['context_instructions'] = [
            '#type' => 'textarea',
            '#title' => $this->t('AI Context Instructions'),
            '#description' => $this->t('Provide additional context or instructions for the AI to consider when processing content.'),
            '#default_value' => $config->get('context_instructions') ?? '',
            '#rows' => 3,
        ];

        $form['merge_instructions'] = [
            '#type' => 'textarea',
            '#title' => $this->t('AI Merge Instructions'),
            '#description' => $this->t('Instructions for the AI when combining or merging data from same products provided by different suppliers into a single enhanced product content.'),
            '#default_value' => $config->get('merge_instructions') ?? '',
            '#rows' => 3,
        ];

        $form['supplier_data'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Supplier Data'),
            '#description' => $this->t('Data coming from the supplier. Use node tokens to include product fields.'),
            '#default_value' => $config->get('supplier_data') ?? '',
            '#rows' => 5,
        ];

        $form['supplier_data_token_tree'] = [
            '#theme' => 'token_tree_link',
            '#token_types' => ['node'],
            '#show_restricted' => TRUE,
            '#global_types' => FALSE,
        ];

        $form['site_data'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Site Data'),
            '#description' => $this->t('Data gathered from our site. Use ai_enhancement tokens for category/brand lists.'),
            '#default_value' => $config->get('site_data') ?? '',
            '#rows' => 4,
        ];

        $form['site_data_token_tree'] = [
            '#theme' => 'token_tree_link',
            '#token_types' => ['ai_enhancement'],
            '#show_restricted' => TRUE,
            '#global_types' => FALSE,
        ];

        $form['prompts'] = [
            '#type' => 'vertical_tabs',
            '#title' => $this->t('AI Prompt Templates'),
            '#description' => $this->t('Configure the default AI prompt templates. These can be overridden per-supplier in the taxonomy term settings.'),
        ];

        // Title Rewrite tab.
        $form['title_group'] = [
            '#type' => 'details',
            '#title' => $this->t('Title Rewrite'),
            '#group' => 'prompts',
        ];

        $form['title_group']['title_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable title rewrite with AI'),
            '#default_value' => $config->get('title_enabled') ?? TRUE,
        ];

        $form['title_group']['title_target_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Target field'),
            '#description' => $this->t('Select the field where the AI-generated title will be stored.'),
            '#options' => $text_fields,
            '#default_value' => $config->get('title_target_field') ?? 'field_ai_title',
        ];

        $form['title_group']['rewrite_title_prompt'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Title Rewrite Prompt'),
            '#default_value' => $config->get('rewrite_title_prompt'),
            '#rows' => 6,
            '#required' => TRUE,
        ];

        // Description Rewrite tab.
        $form['description_group'] = [
            '#type' => 'details',
            '#title' => $this->t('Description Rewrite'),
            '#group' => 'prompts',
        ];

        $form['description_group']['description_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable description rewrite with AI'),
            '#default_value' => $config->get('description_enabled') ?? TRUE,
        ];

        $form['description_group']['description_target_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Target field'),
            '#description' => $this->t('Select the field where the AI-generated description will be stored.'),
            '#options' => $text_fields,
            '#default_value' => $config->get('description_target_field') ?? 'field_ai_description',
        ];

        $form['description_group']['rewrite_description_prompt'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Description Rewrite Prompt'),
            '#default_value' => $config->get('rewrite_description_prompt'),
            '#rows' => 8,
            '#required' => TRUE,
        ];

        // Attribute Extraction tab.
        $form['attributes_group'] = [
            '#type' => 'details',
            '#title' => $this->t('Attribute Extraction'),
            '#group' => 'prompts',
        ];

        $form['attributes_group']['attributes_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable attribute extraction with AI'),
            '#default_value' => $config->get('attributes_enabled') ?? TRUE,
        ];

        $form['attributes_group']['attributes_target_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Target field'),
            '#description' => $this->t('Select the field where the AI-extracted attributes will be stored.'),
            '#options' => $key_value_fields + $text_fields,
            '#default_value' => $config->get('attributes_target_field') ?? 'field_ai_attributes',
        ];

        $form['attributes_group']['extract_attributes_prompt'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Attribute Extraction Prompt'),
            '#default_value' => $config->get('extract_attributes_prompt'),
            '#rows' => 6,
            '#required' => TRUE,
        ];

        // Category Suggestion tab.
        $form['categories_group'] = [
            '#type' => 'details',
            '#title' => $this->t('Category Suggestion'),
            '#group' => 'prompts',
        ];

        $form['categories_group']['categories_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable category suggestion with AI'),
            '#default_value' => $config->get('categories_enabled') ?? TRUE,
        ];

        $form['categories_group']['categories_target_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Target field'),
            '#description' => $this->t('Select the entity reference field where the AI-suggested categories will be stored.'),
            '#options' => $reference_fields,
            '#default_value' => $config->get('categories_target_field') ?? 'field_ai_suggested_categories',
        ];

        $form['categories_group']['suggest_categories_prompt'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Category Suggestion Prompt'),
            '#default_value' => $config->get('suggest_categories_prompt'),
            '#rows' => 10,
            '#required' => TRUE,
        ];

        // Brand Suggestion tab.
        $form['brand_group'] = [
            '#type' => 'details',
            '#title' => $this->t('Brand Suggestion'),
            '#group' => 'prompts',
        ];

        $form['brand_group']['brand_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable brand suggestion with AI'),
            '#default_value' => $config->get('brand_enabled') ?? TRUE,
        ];

        $form['brand_group']['brand_target_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Target field'),
            '#description' => $this->t('Select the entity reference field where the AI-suggested brands will be stored.'),
            '#options' => $reference_fields,
            '#default_value' => $config->get('brand_target_field') ?? 'field_ai_suggested_brand',
        ];

        $form['brand_group']['suggest_brand_prompt'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Brand Suggestion Prompt'),
            '#default_value' => $config->get('suggest_brand_prompt'),
            '#rows' => 10,
            '#required' => TRUE,
        ];

        // Debug settings as a separate collapsible fieldset.
        $form['debug_settings'] = [
            '#type' => 'details',
            '#title' => $this->t('Debug Settings'),
            '#open' => FALSE,
        ];

        $form['debug_settings']['debug_enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable Debug Mode'),
            '#description' => $this->t('When enabled, additional debugging options become available.'),
            '#default_value' => $config->get('debug_enabled') ?? FALSE,
        ];

        $form['debug_settings']['debug_log_prompts'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Log Outgoing Prompts'),
            '#description' => $this->t('Log all prompts sent to the AI provider (visible in Recent log messages).'),
            '#default_value' => $config->get('debug_log_prompts') ?? FALSE,
            '#states' => [
                'visible' => [
                    ':input[name="debug_enabled"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['debug_settings']['debug_log_responses'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Log Incoming Responses'),
            '#description' => $this->t('Log all responses received from the AI provider.'),
            '#default_value' => $config->get('debug_log_responses') ?? FALSE,
            '#states' => [
                'visible' => [
                    ':input[name="debug_enabled"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['debug_settings']['debug_use_mock'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Disable AI Calls and Use Mock Responses'),
            '#description' => $this->t('Instead of calling the AI API, use the original values with a prefix. This allows testing without consuming API tokens.'),
            '#default_value' => $config->get('debug_use_mock') ?? FALSE,
            '#states' => [
                'visible' => [
                    ':input[name="debug_enabled"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['debug_settings']['debug_mock_prefix'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Mock Response Prefix'),
            '#description' => $this->t('Text to prepend to original values when using mock responses.'),
            '#default_value' => $config->get('debug_mock_prefix') ?? '[AI-TEST] ',
            '#size' => 30,
            '#states' => [
                'visible' => [
                    ':input[name="debug_use_mock"]' => ['checked' => TRUE],
                    ':input[name="debug_enabled"]' => ['checked' => TRUE],
                ],
            ],
        ];
        $form['preview'] = [
            '#type' => 'details',
            '#title' => $this->t('Prompt Preview'),
            '#open' => FALSE,
        ];

        // Build the autocomplete deluxe URL for node entities.
        $selection_settings = [
            'target_bundles' => ['supplier_product'],
            'sort' => ['field' => '_none'],
            'auto_create' => FALSE,
            'match_operator' => 'CONTAINS',
        ];
        $target_type = 'node';
        $selection_handler = 'default:node';
        $data = serialize($selection_settings) . $target_type . $selection_handler;
        $selection_settings_key = \Drupal\Component\Utility\Crypt::hmacBase64(
            $data,
            \Drupal\Core\Site\Settings::getHashSalt()
        );

        // Store the selection settings in key_value for the autocomplete route.
        \Drupal::keyValue('entity_autocomplete')->set($selection_settings_key, $selection_settings);

        $autocomplete_url = \Drupal\Core\Url::fromRoute(
            'autocomplete_deluxe.autocomplete',
            [
                'target_type' => $target_type,
                'selection_handler' => $selection_handler,
                'selection_settings_key' => $selection_settings_key,
            ],
            ['absolute' => TRUE]
        )->getInternalPath();

        $form['preview']['preview_product'] = [
            '#type' => 'autocomplete_deluxe',
            '#title' => $this->t('Select Products'),
            '#description' => $this->t('Start typing to search for supplier products. Select multiple products to preview batch mode. Leave empty to use random products based on batch size.'),
            '#autocomplete_deluxe_path' => $autocomplete_url,
            '#target_type' => $target_type,
            '#selection_handler' => $selection_handler,
            '#selection_settings' => $selection_settings,
            '#cardinality' => -1,
            '#multiple' => TRUE,
            '#limit' => 10,
            '#size' => 60,
            '#min_length' => 1,
            '#delimiter' => ',',
            '#new_terms' => FALSE,
            '#not_found_message_allow' => FALSE,
        ];

        $form['preview']['refresh'] = [
            '#type' => 'button',
            '#value' => $this->t('Refresh Preview'),
            '#ajax' => [
                'callback' => '::updatePreview',
                'wrapper' => 'prompt-preview-wrapper',
            ],
            '#limit_validation_errors' => [],
        ];

        $form['preview']['preview_output_wrapper'] = [
            '#type' => 'container',
            '#attributes' => [
                'id' => 'prompt-preview-wrapper',
            ],
        ];

        $form['preview']['preview_output_wrapper']['preview_output'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Outgoing Prompt'),
            '#value' => $this->buildPreviewMarkup($form_state),
            '#rows' => 25,
            '#attributes' => [
                'style' => 'font-family: monospace; background-color: #f5f5f5;',
            ],
        ];

        $form['preview']['submit_prompt'] = [
            '#type' => 'button',
            '#value' => $this->t('Submit Prompt'),
            '#ajax' => [
                'callback' => '::submitPromptToAi',
                'wrapper' => 'prompt-response-wrapper',
                'progress' => [
                    'type' => 'throbber',
                    'message' => $this->t('Submitting to AI...'),
                ],
            ],
            '#limit_validation_errors' => [],
        ];

        $form['preview']['response_wrapper'] = [
            '#type' => 'container',
            '#attributes' => [
                'id' => 'prompt-response-wrapper',
            ],
        ];

        $form['preview']['response_wrapper']['prompt_response'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Prompt Response'),
            '#value' => '',
            '#rows' => 25,
            '#attributes' => [
                'style' => 'font-family: monospace; background-color: #e8f5e9;',
            ],
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * AJAX callback to update the prompt preview.
     */
    public function updatePreview(array &$form, FormStateInterface $form_state): array
    {
        // Update the value in the form element and return the wrapper.
        $form['preview']['preview_output_wrapper']['preview_output']['#value'] = $this->buildPreviewMarkup($form_state);
        return $form['preview']['preview_output_wrapper'];
    }

    /**
     * AJAX callback to submit prompt to AI and display response.
     */
    public function submitPromptToAi(array &$form, FormStateInterface $form_state): array
    {
        $config = $this->config('supplier_products_ai_rewrite.settings');

        // Get the prompt from user input.
        $userInput = $form_state->getUserInput();
        $promptText = $userInput['preview_output'] ?? '';

        if (empty($promptText)) {
            $form['preview']['response_wrapper']['prompt_response']['#value'] = 'Error: No prompt text provided.';
            return $form['preview']['response_wrapper'];
        }

        try {
            // Get AI provider.
            $defaults = $this->aiProviderManager->getDefaultProviderForOperationType('chat');
            if (empty($defaults['provider_id']) || empty($defaults['model_id'])) {
                throw new \RuntimeException('No default AI provider configured for chat operations.');
            }

            $provider = $this->aiProviderManager->createInstance($defaults['provider_id']);
            if (!$provider->isUsable('chat')) {
                throw new \RuntimeException('AI provider is not ready for chat operations.');
            }

            // Build chat messages based on prompt format.
            $prompt_format = $config->get('prompt_format') ?? 'json';
            $chat_messages = [];

            if ($prompt_format === 'plain_text') {
                // Plain text: Send as single user message.
                $chat_messages[] = new ChatMessage('user', $promptText);
            } else {
                // JSON format: Parse and send as system/user messages.
                // Try to extract from JSON format.
                if (preg_match('/messages\s*=\s*(\[.*\])/s', $promptText, $matches)) {
                    $messagesJson = json_decode($matches[1], TRUE);
                    if (is_array($messagesJson)) {
                        foreach ($messagesJson as $msg) {
                            $role = $msg['role'] ?? 'user';
                            $content = $msg['content'] ?? '';
                            if (!empty($content)) {
                                $chat_messages[] = new ChatMessage($role, $content);
                            }
                        }
                    }
                }

                // Fallback: If parsing failed, treat entire text as user message.
                if (empty($chat_messages)) {
                    $chat_messages[] = new ChatMessage('user', $promptText);
                }
            }

            $input = new ChatInput($chat_messages);
            $response = $provider->chat($input, $defaults['model_id']);
            $normalized = $response->getNormalized();

            $result = '';
            if ($normalized instanceof ChatMessage) {
                $result = trim($normalized->getText());
            } else {
                // Handle streamed response.
                $text = '';
                foreach ($normalized as $chat_message) {
                    $text .= $chat_message->getText();
                }
                $result = trim($text);
            }

            $form['preview']['response_wrapper']['prompt_response']['#value'] = $result;
        } catch (\Exception $e) {
            $form['preview']['response_wrapper']['prompt_response']['#value'] = 'Error: ' . $e->getMessage();
        }

        return $form['preview']['response_wrapper'];
    }

    /**
     * Builds the preview markup showing the final prompt as it would be sent to AI.
     */
    protected function buildPreviewMarkup(FormStateInterface $form_state): string
    {
        $config = $this->config('supplier_products_ai_rewrite.settings');
        $node_storage = $this->entityTypeManager->getStorage('node');

        // Get the selected products or random ones based on batch_size.
        $batchSize = (int) ($config->get('batch_size') ?? 1);

        // Get selected products from form state - try both getValue and getUserInput.
        $selectedProducts = $form_state->getValue('preview_product');

        // If no value, try user input (for AJAX callbacks).
        // Autocomplete Deluxe uses a nested structure with 'value_field'.
        if (empty($selectedProducts)) {
            $userInput = $form_state->getUserInput();
            $previewInput = $userInput['preview_product'] ?? NULL;

            // Handle Autocomplete Deluxe's nested value_field structure.
            if (is_array($previewInput) && isset($previewInput['value_field'])) {
                $selectedProducts = $previewInput['value_field'];
            } else {
                $selectedProducts = $previewInput;
            }
        }

        $nodes = [];

        // Handle multiple selected products from #tags entity autocomplete.
        if (!empty($selectedProducts)) {
            // With #tags, the value can be an array of entity references.
            if (is_array($selectedProducts)) {
                foreach ($selectedProducts as $item) {
                    // Each item can be ['target_id' => nid] or just a nid.
                    $nid = is_array($item) ? ($item['target_id'] ?? NULL) : $item;
                    if ($nid) {
                        $node = $node_storage->load($nid);
                        if ($node) {
                            $nodes[] = $node;
                        }
                    }
                }
            } elseif (is_string($selectedProducts)) {
                // Parse comma-separated string with entity labels like "Title (123), Title2 (456)".
                // Extract node IDs from the string.
                preg_match_all('/\((\d+)\)/', $selectedProducts, $matches);
                if (!empty($matches[1])) {
                    foreach ($matches[1] as $nid) {
                        $node = $node_storage->load($nid);
                        if ($node) {
                            $nodes[] = $node;
                        }
                    }
                }
            } else {
                // Single numeric value fallback.
                $node = $node_storage->load($selectedProducts);
                if ($node) {
                    $nodes[] = $node;
                }
            }
        }

        // If no products selected, get random ones based on batch_size.
        if (empty($nodes)) {
            $query = $node_storage->getQuery()
                ->condition('type', 'supplier_product')
                ->condition('status', 1)
                ->accessCheck(FALSE)
                ->range(0, $batchSize);
            $nids = $query->execute();
            if ($nids) {
                $nodes = array_values($node_storage->loadMultiple($nids));
            }
        }

        if (empty($nodes)) {
            return $this->t('No supplier products found. Create a supplier product to preview prompts.');
        }

        // Determine which tasks are enabled.
        $tasks = [];
        if ($config->get('title_enabled')) {
            $tasks[] = 'title';
        }
        if ($config->get('description_enabled')) {
            $tasks[] = 'description';
        }
        if ($config->get('attributes_enabled')) {
            $tasks[] = 'attributes';
        }
        if ($config->get('categories_enabled')) {
            $tasks[] = 'categorize';
        }
        if ($config->get('brand_enabled')) {
            $tasks[] = 'brand';
        }

        // Use the actual PromptBuilder service to generate the prompt.
        if (count($nodes) === 1) {
            // Single product - use buildUnifiedPromptMessages.
            $messages = $this->promptBuilder->buildUnifiedPromptMessages($nodes[0], $tasks);
        } else {
            // Multiple products - use buildBatchPrompt.
            $batchItems = [];
            foreach ($nodes as $node) {
                $batchItems[] = [
                    'node' => $node,
                    'tasks' => $tasks,
                    'supplier' => $node->hasField('field_supplier') && !$node->get('field_supplier')->isEmpty()
                        ? $node->get('field_supplier')->entity
                        : NULL,
                ];
            }
            $messages = $this->promptBuilder->buildBatchPrompt($batchItems);
        }

        // Format display based on prompt_format setting.
        $prompt_format = $config->get('prompt_format') ?? 'json';

        if ($prompt_format === 'plain_text') {
            // Plain text: Show combined content as raw text (exactly what gets sent).
            $combined_content = '';
            if (!empty($messages['system'])) {
                $combined_content = $messages['system'] . "\n\n---\n\n";
            }
            $combined_content .= $messages['user'];

            return $combined_content;
        } else {
            // JSON format: Show separate system and user messages as JSON array.
            $formattedMessages = [
                [
                    'role' => 'system',
                    'content' => $messages['system'],
                ],
                [
                    'role' => 'user',
                    'content' => $messages['user'],
                ],
            ];

            return "messages = " . json_encode($formattedMessages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        }
    }

    /**
     * Gets a list of taxonomy terms from a vocabulary.
     */
    protected function getTaxonomyList($term_storage, string $vocabulary): string
    {
        $terms = $term_storage->loadTree($vocabulary, 0, NULL, TRUE);
        $names = [];
        foreach ($terms as $term) {
            $indent = $term->depth > 0 ? str_repeat('- ', $term->depth) : '';
            $names[] = $indent . $term->label();
        }
        return implode("\n", $names);
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        $this->config('supplier_products_ai_rewrite.settings')
            // Global settings.
            ->set('enabled', $form_state->getValue('enabled'))
            ->set('enhance_on_create', $form_state->getValue('enhance_on_create'))
            ->set('enhance_on_merge', $form_state->getValue('enhance_on_merge'))
            ->set('role_description', $form_state->getValue('role_description'))
            ->set('task_description', $form_state->getValue('task_description'))
            ->set('context_instructions', $form_state->getValue('context_instructions'))
            ->set('merge_instructions', $form_state->getValue('merge_instructions'))
            ->set('supplier_data', $form_state->getValue('supplier_data'))
            ->set('site_data', $form_state->getValue('site_data'))
            // Title settings.
            ->set('title_enabled', $form_state->getValue('title_enabled'))
            ->set('title_target_field', $form_state->getValue('title_target_field'))
            ->set('rewrite_title_prompt', $form_state->getValue('rewrite_title_prompt'))
            // Description settings.
            ->set('description_enabled', $form_state->getValue('description_enabled'))
            ->set('description_target_field', $form_state->getValue('description_target_field'))
            ->set('rewrite_description_prompt', $form_state->getValue('rewrite_description_prompt'))
            // Attributes settings.
            ->set('attributes_enabled', $form_state->getValue('attributes_enabled'))
            ->set('attributes_target_field', $form_state->getValue('attributes_target_field'))
            ->set('extract_attributes_prompt', $form_state->getValue('extract_attributes_prompt'))
            // Categories settings.
            ->set('categories_enabled', $form_state->getValue('categories_enabled'))
            ->set('categories_target_field', $form_state->getValue('categories_target_field'))
            ->set('suggest_categories_prompt', $form_state->getValue('suggest_categories_prompt'))
            // Brand settings.
            ->set('brand_enabled', $form_state->getValue('brand_enabled'))
            ->set('brand_target_field', $form_state->getValue('brand_target_field'))
            ->set('suggest_brand_prompt', $form_state->getValue('suggest_brand_prompt'))
            // Debug settings.
            ->set('debug_enabled', $form_state->getValue('debug_enabled'))
            ->set('debug_log_prompts', $form_state->getValue('debug_log_prompts'))
            ->set('debug_log_responses', $form_state->getValue('debug_log_responses'))
            ->set('debug_use_mock', $form_state->getValue('debug_use_mock'))
            ->set('debug_mock_prefix', $form_state->getValue('debug_mock_prefix'))
            // Batch settings.
            ->set('batch_size', $form_state->getValue('batch_size'))
            ->set('prompt_format', $form_state->getValue('prompt_format'))
            ->save();

        parent::submitForm($form, $form_state);
    }
}
