<?php

namespace Drupal\supplier_products_ai_rewrite\Form;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Form for managing AI cache import/export and configuration.
 */
class CacheManagementForm extends FormBase
{
    protected FileSystemInterface $fileSystem;
    protected StreamWrapperManagerInterface $streamWrapperManager;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container): static
    {
        $instance = new static();
        $instance->fileSystem = $container->get('file_system');
        $instance->streamWrapperManager = $container->get('stream_wrapper_manager');
        $instance->setConfigFactory($container->get('config.factory'));
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId(): string
    {
        return 'supplier_products_ai_rewrite_cache_management';
    }

    /**
     * Gets the cache directory based on config.
     */
    protected function getCacheDirectory(): string
    {
        $config = $this->configFactory()->get('supplier_products_ai_rewrite.settings');
        $storage = $config->get('cache_storage') ?? 'public';
        return $storage . '://aicache';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state): array
    {
        $stats = $this->getCacheStatistics();

        // Cache Statistics (styled cards above tabs).
        $form['statistics'] = [
            '#type' => 'inline_template',
            '#template' => '
                <style>
                    .ai-cache-stats { display: flex; gap: 16px; margin-bottom: 24px; flex-wrap: wrap; }
                    .ai-cache-stat-card { 
                        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                        border: 1px solid #dee2e6;
                        border-radius: 8px;
                        padding: 16px 24px;
                        text-align: center;
                        min-width: 140px;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.05);
                    }
                    .ai-cache-stat-card .stat-value {
                        font-size: 28px;
                        font-weight: 700;
                        color: #0073aa;
                        line-height: 1.2;
                    }
                    .ai-cache-stat-card .stat-label {
                        font-size: 12px;
                        text-transform: uppercase;
                        color: #6c757d;
                        letter-spacing: 0.5px;
                        margin-top: 4px;
                    }
                </style>
                <div class="ai-cache-stats">
                    <div class="ai-cache-stat-card">
                        <div class="stat-value">{{ products }}</div>
                        <div class="stat-label">{{ "Products"|t }}</div>
                    </div>
                    <div class="ai-cache-stat-card">
                        <div class="stat-value">{{ files }}</div>
                        <div class="stat-label">{{ "Cache Files"|t }}</div>
                    </div>
                    <div class="ai-cache-stat-card">
                        <div class="stat-value">{{ size }}</div>
                        <div class="stat-label">{{ "Total Size"|t }}</div>
                    </div>
                    <div class="ai-cache-stat-card">
                        <div class="stat-value">{{ contexts }}</div>
                        <div class="stat-label">{{ "Contexts"|t }}</div>
                    </div>
                </div>
            ',
            '#context' => [
                'products' => $stats['product_count'],
                'files' => $stats['file_count'],
                'size' => \Drupal\Core\StringTranslation\ByteSizeMarkup::create($stats['total_size']),
                'contexts' => $stats['context_count'],
            ],
        ];

        // Vertical tabs container.
        $form['tabs'] = [
            '#type' => 'vertical_tabs',
            '#default_tab' => 'edit-import',
        ];

        // Storage Settings Tab.
        $form['storage'] = [
            '#type' => 'details',
            '#title' => $this->t('Storage'),
            '#group' => 'tabs',
            '#weight' => 2,
        ];

        $privateWritable = $this->isPrivateDirectoryWritable();
        $currentStorage = $this->configFactory()->get('supplier_products_ai_rewrite.settings')->get('cache_storage') ?? 'public';

        $form['storage']['cache_storage'] = [
            '#type' => 'radios',
            '#title' => $this->t('Cache Storage Location'),
            '#options' => [
                'public' => $this->t('Public files (public://)'),
                'private' => $this->t('Private files (private://)'),
            ],
            '#default_value' => $currentStorage,
        ];

        if (!$privateWritable) {
            $form['storage']['cache_storage']['private']['#disabled'] = TRUE;
            $form['storage']['private_warning'] = [
                '#type' => 'item',
                '#markup' => '<div class="messages messages--warning">' . $this->t('Private file system is not configured or not writable. Configure it in settings.php to enable this option.') . '</div>',
            ];
        }

        $form['storage']['old_cache_action'] = [
            '#type' => 'radios',
            '#title' => $this->t('When changing location, existing cache files:'),
            '#options' => [
                'move' => $this->t('Move to new location'),
                'keep' => $this->t('Keep in old location (will be ignored)'),
                'delete' => $this->t('Delete from old location'),
            ],
            '#default_value' => 'move',
            '#description' => $this->t('Choose what to do with cache files in the current location when switching.'),
        ];

        $form['storage']['new_location_strategy'] = [
            '#type' => 'radios',
            '#title' => $this->t('If files exist in new location:'),
            '#options' => [
                'merge' => $this->t('Merge (keep existing, only add new files)'),
                'overwrite' => $this->t('Overwrite (replace conflicting files)'),
                'clear' => $this->t('Clear first (delete all before moving)'),
            ],
            '#default_value' => 'merge',
            '#states' => [
                'visible' => [
                    ':input[name="old_cache_action"]' => ['value' => 'move'],
                ],
            ],
        ];

        $form['storage']['save_storage'] = [
            '#type' => 'submit',
            '#value' => $this->t('Save Storage Setting'),
            '#submit' => ['::saveStorageSetting'],
        ];

        // Export Tab.
        $form['export'] = [
            '#type' => 'details',
            '#title' => $this->t('Export'),
            '#group' => 'tabs',
            '#weight' => 1,
        ];

        $form['export']['export_options'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('What to export'),
            '#options' => [
                'cache' => $this->t('AI Cache (@count files, @size)', [
                    '@count' => $stats['file_count'],
                    '@size' => \Drupal\Core\StringTranslation\ByteSizeMarkup::create($stats['total_size']),
                ]),
                'config' => $this->t('Configuration (prompts and settings)'),
            ],
            '#default_value' => ['cache', 'config'],
        ];

        $form['export']['export_submit'] = [
            '#type' => 'submit',
            '#value' => $this->t('Export'),
            '#submit' => ['::exportCombined'],
        ];

        // Import Tab.
        $form['import'] = [
            '#type' => 'details',
            '#title' => $this->t('Import'),
            '#group' => 'tabs',
            '#weight' => 0,
        ];

        $form['import']['import_file'] = [
            '#type' => 'file',
            '#title' => $this->t('Upload Archive'),
            '#description' => $this->t('Upload a .zip archive containing cache and/or config JSON.'),
        ];

        $form['import']['import_options'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('What to import'),
            '#options' => [
                'cache' => $this->t('AI Cache'),
                'config' => $this->t('Configuration'),
            ],
            '#default_value' => ['cache', 'config'],
        ];

        $form['import']['cache_strategy'] = [
            '#type' => 'radios',
            '#title' => $this->t('Existing cache handling'),
            '#options' => [
                'merge' => $this->t('Merge (keep existing, only add new files)'),
                'overwrite' => $this->t('Overwrite (replace conflicting files)'),
                'replace' => $this->t('Replace (delete all existing, import new)'),
            ],
            '#default_value' => 'merge',
            '#states' => [
                'visible' => [
                    ':input[name="import_options[cache]"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['import']['import_submit'] = [
            '#type' => 'submit',
            '#value' => $this->t('Import'),
            '#submit' => ['::importCombined'],
        ];

        // Clear Cache Tab.
        $form['clear'] = [
            '#type' => 'details',
            '#title' => $this->t('Clear Cache'),
            '#group' => 'tabs',
            '#weight' => 3,
        ];

        $form['clear']['warning'] = [
            '#type' => 'item',
            '#markup' => '<div class="messages messages--warning">' . $this->t('This action cannot be undone. All cached AI responses will be permanently deleted.') . '</div>',
        ];

        $form['clear']['clear_cache'] = [
            '#type' => 'submit',
            '#value' => $this->t('Clear All Cache'),
            '#submit' => ['::clearCache'],
            '#attributes' => ['class' => ['button--danger']],
            '#disabled' => $stats['file_count'] === 0,
        ];

        return $form;
    }

    /**
     * Checks if private directory is writable.
     */
    protected function isPrivateDirectoryWritable(): bool
    {
        $privateDir = $this->fileSystem->realpath('private://');
        if (!$privateDir) {
            return FALSE;
        }
        return is_dir($privateDir) && is_writable($privateDir);
    }

    /**
     * Save storage setting handler.
     */
    public function saveStorageSetting(array &$form, FormStateInterface $form_state): void
    {
        $newStorage = $form_state->getValue('cache_storage');
        $oldCacheAction = $form_state->getValue('old_cache_action');
        $config = $this->configFactory()->getEditable('supplier_products_ai_rewrite.settings');
        $currentStorage = $config->get('cache_storage') ?? 'public';

        // If storage hasn't changed, just save and return.
        if ($newStorage === $currentStorage) {
            $this->messenger()->addStatus($this->t('Storage location unchanged.'));
            return;
        }

        $oldCacheDir = $currentStorage . '://aicache';
        $newCacheDir = $newStorage . '://aicache';
        $oldRealPath = $this->fileSystem->realpath($oldCacheDir);

        // Handle old cache files based on selected action.
        if ($oldRealPath && is_dir($oldRealPath)) {
            switch ($oldCacheAction) {
                case 'move':
                    $newLocationStrategy = $form_state->getValue('new_location_strategy');

                    // Prepare new directory.
                    $this->fileSystem->prepareDirectory($newCacheDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
                    $newRealPath = $this->fileSystem->realpath($newCacheDir);

                    // Handle existing files in new location based on strategy.
                    if ($newRealPath && is_dir($newRealPath) && $newLocationStrategy === 'clear') {
                        $this->fileSystem->deleteRecursive($newRealPath);
                        mkdir($newRealPath, 0755, TRUE);
                        $this->messenger()->addStatus($this->t('Cleared existing files in new location.'));
                    }

                    // Copy files to new location with merge/overwrite logic.
                    $skipExisting = ($newLocationStrategy === 'merge');
                    $this->recursiveCopy($oldRealPath, $newRealPath, $skipExisting);

                    // Delete old directory.
                    $this->fileSystem->deleteRecursive($oldRealPath);

                    $this->messenger()->addStatus($this->t('Cache files moved to new location.'));
                    break;

                case 'delete':
                    $this->fileSystem->deleteRecursive($oldRealPath);
                    $this->messenger()->addStatus($this->t('Old cache files deleted.'));
                    break;

                case 'keep':
                default:
                    $this->messenger()->addWarning($this->t('Old cache files kept at @path but will no longer be used.', ['@path' => $oldCacheDir]));
                    break;
            }
        }

        // Save new storage setting.
        $config->set('cache_storage', $newStorage)->save();
        $this->messenger()->addStatus($this->t('Cache storage changed to @storage.', ['@storage' => $newStorage . '://']));
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        // Main submit is not used - each button has its own handler.
    }

    /**
     * Gets cache statistics.
     */
    protected function getCacheStatistics(): array
    {
        $stats = [
            'file_count' => 0,
            'total_size' => 0,
            'context_count' => 0,
            'product_count' => 0,
        ];

        $cacheDir = $this->fileSystem->realpath($this->getCacheDirectory());
        if (!$cacheDir || !is_dir($cacheDir)) {
            return $stats;
        }

        // Count context directories.
        $contexts = glob($cacheDir . '/*', GLOB_ONLYDIR);
        $stats['context_count'] = count($contexts);

        // Count all files and sum sizes, track unique product hashes.
        $productHashes = [];
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($cacheDir, \FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $stats['file_count']++;
                $stats['total_size'] += $file->getSize();

                // Extract product hash from filename (e.g., "abc123def456.json" -> "abc123def456")
                $filename = pathinfo($file->getFilename(), PATHINFO_FILENAME);
                $productHashes[$filename] = TRUE;
            }
        }

        $stats['product_count'] = count($productHashes);

        return $stats;
    }

    /**
     * Combined export handler.
     */
    public function exportCombined(array &$form, FormStateInterface $form_state): void
    {
        $options = array_filter($form_state->getValue('export_options'));

        if (empty($options)) {
            $this->messenger()->addError($this->t('Please select at least one option to export.'));
            return;
        }

        try {
            $tempFile = $this->fileSystem->tempnam('temporary://', 'ai_export_') . '.zip';
            $realTempFile = $this->fileSystem->realpath($tempFile);

            $zip = new \ZipArchive();
            if ($zip->open($realTempFile, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== TRUE) {
                $this->messenger()->addError($this->t('Failed to create archive.'));
                return;
            }

            // Export cache if selected.
            if (!empty($options['cache'])) {
                $cacheDir = $this->fileSystem->realpath($this->getCacheDirectory());
                if ($cacheDir && is_dir($cacheDir)) {
                    $iterator = new \RecursiveIteratorIterator(
                        new \RecursiveDirectoryIterator($cacheDir, \FilesystemIterator::SKIP_DOTS),
                        \RecursiveIteratorIterator::SELF_FIRST
                    );

                    foreach ($iterator as $file) {
                        $relativePath = 'cache/' . str_replace($cacheDir . '/', '', $file->getPathname());
                        if ($file->isDir()) {
                            $zip->addEmptyDir($relativePath);
                        } else {
                            $zip->addFile($file->getPathname(), $relativePath);
                        }
                    }
                }
            }

            // Export config if selected.
            if (!empty($options['config'])) {
                $config = $this->configFactory()->get('supplier_products_ai_rewrite.settings');
                $data = $config->getRawData();
                unset($data['_core']);
                $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                $zip->addFromString('config.json', $json);
            }

            $zip->close();

            $response = new BinaryFileResponse($realTempFile);
            $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, 'ai_export_' . date('Y-m-d_His') . '.zip');
            $response->deleteFileAfterSend(TRUE);
            $form_state->setResponse($response);
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Export failed: @message', ['@message' => $e->getMessage()]));
        }
    }

    /**
     * Combined import handler.
     */
    public function importCombined(array &$form, FormStateInterface $form_state): void
    {
        $files = $this->getRequest()->files->get('files', []);
        $uploadedFile = $files['import_file'] ?? NULL;

        if (!$uploadedFile || !$uploadedFile->isValid()) {
            $this->messenger()->addError($this->t('Please upload a valid zip file.'));
            return;
        }

        $options = array_filter($form_state->getValue('import_options'));
        if (empty($options)) {
            $this->messenger()->addError($this->t('Please select at least one option to import.'));
            return;
        }

        $tempPath = $uploadedFile->getRealPath();
        $zip = new \ZipArchive();
        if ($zip->open($tempPath) !== TRUE) {
            $this->messenger()->addError($this->t('Failed to open the archive.'));
            return;
        }

        // Extract to temp directory first.
        $tempDir = $this->fileSystem->tempnam('temporary://', 'ai_import_');
        unlink($this->fileSystem->realpath($tempDir));
        mkdir($this->fileSystem->realpath('temporary://') . '/' . basename($tempDir));
        $realTempDir = $this->fileSystem->realpath('temporary://') . '/' . basename($tempDir);
        $zip->extractTo($realTempDir);
        $zip->close();

        // Import config if selected and exists.
        if (!empty($options['config']) && file_exists($realTempDir . '/config.json')) {
            $content = file_get_contents($realTempDir . '/config.json');
            $data = json_decode($content, TRUE);
            if (is_array($data)) {
                $config = $this->configFactory()->getEditable('supplier_products_ai_rewrite.settings');
                foreach ($data as $key => $value) {
                    $config->set($key, $value);
                }
                $config->save();
                $this->messenger()->addStatus($this->t('Configuration imported successfully.'));
            }
        }

        // Import cache if selected and exists.
        if (!empty($options['cache']) && is_dir($realTempDir . '/cache')) {
            $cacheStrategy = $form_state->getValue('cache_strategy');

            // If replace, clear existing cache first.
            if ($cacheStrategy === 'replace') {
                $cacheDir = $this->fileSystem->realpath($this->getCacheDirectory());
                if ($cacheDir && is_dir($cacheDir)) {
                    $this->fileSystem->deleteRecursive($cacheDir);
                }
            }

            // Ensure cache directory exists.
            $cacheDir = $this->getCacheDirectory();
            $this->fileSystem->prepareDirectory($cacheDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
            $realCacheDir = $this->fileSystem->realpath($cacheDir);

            // Copy files from import with appropriate strategy.
            $skipExisting = ($cacheStrategy === 'merge');
            $this->recursiveCopy($realTempDir . '/cache', $realCacheDir, $skipExisting);
            $this->messenger()->addStatus($this->t('Cache imported successfully.'));
        }

        // Cleanup temp directory.
        $this->fileSystem->deleteRecursive($realTempDir);
    }

    /**
     * Recursively copies files from source to destination.
     *
     * @param string $source
     *   Source directory path.
     * @param string $dest
     *   Destination directory path.
     * @param bool $skipExisting
     *   If TRUE, skip files that already exist in destination.
     *
     * @throws \RuntimeException
     *   If a critical file operation fails.
     */
    protected function recursiveCopy(string $source, string $dest, bool $skipExisting = FALSE): void
    {
        if (!is_dir($source)) {
            throw new \RuntimeException("Source directory does not exist: $source");
        }

        if (!is_dir($dest)) {
            if (!@mkdir($dest, 0755, TRUE)) {
                throw new \RuntimeException("Failed to create destination directory: $dest");
            }
        }

        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($source, \FilesystemIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            $relativePath = str_replace($source . '/', '', $file->getPathname());
            $destPath = $dest . '/' . $relativePath;

            if ($file->isDir()) {
                if (!is_dir($destPath) && !@mkdir($destPath, 0755, TRUE)) {
                    throw new \RuntimeException("Failed to create directory: $destPath");
                }
            } else {
                // Skip if file exists and we're in merge mode.
                if ($skipExisting && file_exists($destPath)) {
                    continue;
                }
                if (!@copy($file->getPathname(), $destPath)) {
                    throw new \RuntimeException("Failed to copy file: {$file->getPathname()} to $destPath");
                }
            }
        }
    }

    /**
     * Clear cache submit handler.
     */
    public function clearCache(array &$form, FormStateInterface $form_state): void
    {
        $cacheDir = $this->fileSystem->realpath($this->getCacheDirectory());

        if (!$cacheDir || !is_dir($cacheDir)) {
            $this->messenger()->addWarning($this->t('No cache directory found.'));
            return;
        }

        try {
            $this->fileSystem->deleteRecursive($cacheDir);
            $this->messenger()->addStatus($this->t('All AI cache has been cleared.'));
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Failed to clear cache: @message', ['@message' => $e->getMessage()]));
        }
    }
}
