<?php

namespace Drupal\supplier_products_ai_rewrite\EventSubscriber;

use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Adds AI view components when the supplier product management view is saved.
 */
class ViewConfigSubscriber implements EventSubscriberInterface
{

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * Constructs a ViewConfigSubscriber object.
     *
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     */
    public function __construct(ConfigFactoryInterface $config_factory)
    {
        $this->configFactory = $config_factory;
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents(): array
    {
        return [
            ConfigEvents::SAVE => ['onConfigSave', -100],
        ];
    }

    /**
     * Responds to config save events.
     *
     * When the supplier_product_management view is saved (e.g., during profile
     * config import), add the AI enhancement components if not present.
     *
     * @param \Drupal\Core\Config\ConfigCrudEvent $event
     *   The config event.
     */
    public function onConfigSave(ConfigCrudEvent $event): void
    {
        $config = $event->getConfig();

        // Only act on the supplier product management view.
        if ($config->getName() !== 'views.view.supplier_product_management') {
            return;
        }

        // Check if AI columns are already present.
        $fields = $config->get('display.default.display_options.fields') ?? [];
        if (isset($fields['ai_enhancement_status'])) {
            // Already has the AI status column, skip.
            return;
        }

        // Load the helper function from .install file and add components.
        \Drupal::moduleHandler()->loadInclude('supplier_products_ai_rewrite', 'install');
        if (function_exists('_supplier_products_ai_rewrite_add_view_components')) {
            _supplier_products_ai_rewrite_add_view_components();
        }
    }
}
