<?php

namespace Drupal\supplier_products_ai_rewrite\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\node\NodeInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for AI enhancement operations.
 */
class AIEnhanceController extends ControllerBase
{

    /**
     * The queue worker manager.
     *
     * @var \Drupal\Core\Queue\QueueWorkerManagerInterface
     */
    protected QueueWorkerManagerInterface $queueWorkerManager;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected LoggerInterface $logger;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container): static
    {
        $instance = parent::create($container);
        $instance->queueWorkerManager = $container->get('plugin.manager.queue_worker');
        $instance->logger = $container->get('logger.factory')->get('supplier_products_ai_rewrite');
        return $instance;
    }

    /**
     * Enhances a node with AI.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The node to enhance.
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     *   Redirect back to the node page.
     */
    public function enhance(NodeInterface $node): RedirectResponse
    {
        if ($node->bundle() !== 'supplier_product') {
            $this->messenger()->addError($this->t('This action is only available for supplier products.'));
            return new RedirectResponse($node->toUrl()->toString());
        }

        // Determine which tasks to process based on available content.
        $tasks = [];

        if (!$node->get('field_original_product_title')->isEmpty()) {
            $tasks[] = 'title';
        }

        if (!$node->get('field_supplier_product_descripti')->isEmpty()) {
            $tasks[] = 'description';
            $tasks[] = 'attributes';
            $tasks[] = 'categorize';
        }

        if (empty($tasks)) {
            $this->messenger()->addWarning($this->t('No content available to enhance. Please add original title or description first.'));
            return new RedirectResponse($node->toUrl()->toString());
        }

        // Get current signatures.
        $title_sig = $node->get('field_title_signature')->value;
        $desc_sig = $node->get('field_description_signature')->value;

        // Process immediately using the queue worker.
        $queue_worker = $this->queueWorkerManager->createInstance('ai_content_enhancer');

        $item = [
            'nid' => $node->id(),
            'tasks' => $tasks,
            'title_sig' => $title_sig,
            'desc_sig' => $desc_sig,
        ];

        try {
            $queue_worker->processItem($item);

            // Reload the node to see the changes.
            $reloaded_node = $this->entityTypeManager()->getStorage('node')->load($node->id());
            if (!$reloaded_node instanceof NodeInterface) {
                $this->messenger()->addError($this->t('Unable to reload the product after enhancement.'));
                return new RedirectResponse($node->toUrl()->toString());
            }
            $node = $reloaded_node;

            // Check if AI fields were populated using config-based field names.
            $has_changes = FALSE;
            $changes = [];
            $config = $this->config('supplier_products_ai_rewrite.settings');

            $titleField = $config->get('title_target_field') ?? 'field_ai_title';
            if ($node->hasField($titleField) && !$node->get($titleField)->isEmpty()) {
                $changes[] = 'title';
                $has_changes = TRUE;
            }

            $descField = $config->get('description_target_field') ?? 'field_ai_description';
            if ($node->hasField($descField) && !$node->get($descField)->isEmpty()) {
                $changes[] = 'description';
                $has_changes = TRUE;
            }

            $attrField = $config->get('attributes_target_field') ?? 'field_ai_attributes';
            if ($node->hasField($attrField) && !$node->get($attrField)->isEmpty()) {
                $changes[] = 'attributes';
                $has_changes = TRUE;
            }

            $catField = $config->get('categories_target_field') ?? 'field_ai_suggested_categories';
            if ($node->hasField($catField) && !$node->get($catField)->isEmpty()) {
                $changes[] = 'categories';
                $has_changes = TRUE;
            }

            if ($has_changes) {
                $this->messenger()->addStatus($this->t('Product enhanced with AI: @changes', [
                    '@changes' => implode(', ', $changes)
                ]));
            } else {
                $this->messenger()->addWarning($this->t('AI enhancement completed but no content was generated. Check the logs for details.'));
            }
        } catch (\Exception $e) {
            $this->logger->error('AI enhancement failed: @message. Trace: @trace', [
                '@message' => $e->getMessage(),
                '@trace' => $e->getTraceAsString(),
            ]);
            $this->messenger()->addError($this->t('AI enhancement failed: @message', ['@message' => $e->getMessage()]));
        }

        return new RedirectResponse($node->toUrl()->toString());
    }
}
