<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\feeds\FeedInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to provide a visual progress bar for feed imports.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("feed_progress")
 */
class FeedProgress extends FieldPluginBase
{
    use FeedAutoRefreshTrait;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * Constructs a FeedProgress object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function clickSortable()
    {
        // This field is not sortable.
        return FALSE;
    }

    /**
     * {@inheritdoc}
     */
    protected function defineOptions()
    {
        $options = parent::defineOptions();
        $options['auto_refresh'] = ['default' => TRUE];
        $options['refresh_interval'] = ['default' => 5];
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function buildOptionsForm(&$form, FormStateInterface $form_state)
    {
        parent::buildOptionsForm($form, $form_state);

        $form['auto_refresh'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Auto-refresh progress'),
            '#default_value' => $this->options['auto_refresh'],
            '#description' => $this->t('Periodically re-fetch the progress via AJAX to show live import progress.'),
        ];

        $form['refresh_interval'] = [
            '#type' => 'number',
            '#title' => $this->t('Refresh interval (seconds)'),
            '#min' => $this->minimumRefreshInterval,
            '#default_value' => $this->options['refresh_interval'],
            '#description' => $this->t('How often to refresh the progress bar when auto-refresh is enabled.'),
            '#states' => [
                'visible' => [
                    ':input[name="options[auto_refresh]"]' => ['checked' => TRUE],
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $feed = $this->getEntity($values);

        if (!$feed instanceof FeedInterface) {
            return '';
        }

        $build = $this->buildProgressBar($feed);
        return $this->wrapWithAutoRefreshContainer($build, $feed);
    }

    /**
     * Builds the progress bar render array.
     */
    protected function buildProgressBar(FeedInterface $feed): array
    {
        // Calculate progress percentage
        $percentage = 0;

        // Check if feed is currently importing
        try {
            // Only show progress if actively importing (locked)
            if ($feed->isLocked()) {
                // Get current import progress from the feed state
                if (method_exists($feed, 'getState')) {
                    $state = $feed->getState(\Drupal\feeds\StateInterface::PROCESS);
                    if ($state) {
                        $total = $state->total ?? 0;

                        // If state total is 0, try using feed item count
                        if ($total == 0) {
                            $total = (int) $feed->getItemCount();
                        }

                        if ($total > 0) {
                            // Calculate progress based on processed items
                            $processed = ($state->created ?? 0) + ($state->updated ?? 0) +
                                ($state->deleted ?? 0) + ($state->skipped ?? 0) +
                                ($state->failed ?? 0);
                            $percentage = min(100, max(0, round(($processed / $total) * 100)));
                        } else {
                            // Fallback to state progress property (0.0 to 1.0)
                            $progress_ratio = $state->progress ?? 0;
                            $percentage = min(100, max(0, round($progress_ratio * 100)));
                        }
                    }
                }
            }
            // If not locked/importing, show 0%
        } catch (\Exception $e) {
            // If there's an error getting state, default to 0%
            $percentage = 0;
        }

        // Build the progress bar HTML
        return [
            '#type' => 'inline_template',
            '#template' => '<div class="feed-progress-wrapper" data-percentage="{{ percentage }}"><div class="percentage-label" data-percentage="{{ percentage }}">{{ percentage }}%</div><div class="progress-bar-container"><div class="progress-bar" style="width: {{ percentage }}%;"></div></div></div>',
            '#context' => [
                'percentage' => $percentage,
            ],
            '#attached' => [
                'library' => [
                    'supplier_products/feed-progress',
                ],
            ],
        ];
    }
}
