<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to provide the feed processor mode (Create/Update).
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("feed_processor_mode")
 */
class FeedProcessorMode extends FieldPluginBase
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * Constructs a FeedProcessorMode object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     */
    public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $feed = $this->getEntity($values);

        if (!$feed) {
            return '';
        }

        // Get the feed type entity
        $feed_type_id = $feed->bundle();
        try {
            $feed_type = $this->entityTypeManager->getStorage('feeds_feed_type')->load($feed_type_id);

            if (!$feed_type) {
                return '';
            }

            // Get the processor configuration
            /** @var \Drupal\feeds\Plugin\PluginInterface $processor */
            $processor = $feed_type->getProcessor();
            $configuration = $processor->getConfiguration();

            // Check insert_new setting
            // 0 = Do not insert new content items
            // 1 = Insert new content items
            $insert_new = $configuration['insert_new'] ?? 1;

            // Check update_existing setting
            // 0 = Do not update existing content items
            // 1 = Replace existing content items
            // 2 = Update existing content items
            $update_existing = $configuration['update_existing'] ?? 0;

            // If "Insert new content items" is selected, always show "Create"
            if ($insert_new == 1) {
                return 'Create';
            }

            // If "Do not insert new content items" is selected, check update settings
            if ($insert_new == 0) {
                // If update setting is either replace (1) or update (2), show "Update"
                if ($update_existing == 1 || $update_existing == 2) {
                    return 'Update';
                }
                // If both "do not insert" and "do not update", show nothing
                return '';
            }

            return '';
        } catch (\Exception $e) {
            return '';
        }
    }
}
