<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\feeds\FeedInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to provide an auto-refreshing items imported count.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("feed_items_imported")
 */
class FeedItemsImported extends FieldPluginBase
{
    use FeedAutoRefreshTrait;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * Constructs a FeedItemsImported object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function clickSortable()
    {
        // This field is not sortable.
        return FALSE;
    }

    /**
     * {@inheritdoc}
     */
    protected function defineOptions()
    {
        $options = parent::defineOptions();
        $options['auto_refresh'] = ['default' => TRUE];
        $options['refresh_interval'] = ['default' => 5];
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function buildOptionsForm(&$form, FormStateInterface $form_state)
    {
        parent::buildOptionsForm($form, $form_state);

        $form['auto_refresh'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Auto-refresh count'),
            '#default_value' => $this->options['auto_refresh'],
            '#description' => $this->t('Periodically re-fetch the count via AJAX to show live import progress.'),
        ];

        $form['refresh_interval'] = [
            '#type' => 'number',
            '#title' => $this->t('Refresh interval (seconds)'),
            '#min' => $this->minimumRefreshInterval,
            '#default_value' => $this->options['refresh_interval'],
            '#description' => $this->t('How often to refresh the count when auto-refresh is enabled.'),
            '#states' => [
                'visible' => [
                    ':input[name="options[auto_refresh]"]' => ['checked' => TRUE],
                ],
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $feed = $this->getEntity($values);

        if (!$feed instanceof FeedInterface) {
            return '';
        }

        $build = $this->buildItemsImported($feed);
        return $this->wrapWithAutoRefreshContainer($build, $feed);
    }

    /**
     * Builds the items imported render array.
     */
    protected function buildItemsImported(FeedInterface $feed): array
    {
        $count = (int) $feed->getItemCount();

        return [
            '#markup' => $count,
        ];
    }
}
