<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\feeds\FeedInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to display combined last import and next import times with auto-refresh.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("feed_import_times")
 */
class FeedImportTimes extends FieldPluginBase
{
    use FeedAutoRefreshTrait;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The date formatter service.
     *
     * @var \Drupal\Core\Datetime\DateFormatterInterface
     */
    protected $dateFormatter;

    /**
     * Constructs a FeedImportTimes object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
     *   The date formatter service.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager,
        DateFormatterInterface $date_formatter
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
        $this->dateFormatter = $date_formatter;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager'),
            $container->get('date.formatter')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function clickSortable()
    {
        // This field is not sortable.
        return FALSE;
    }

    /**
     * {@inheritdoc}
     */
    protected function defineOptions()
    {
        $options = parent::defineOptions();
        $options['auto_refresh'] = ['default' => TRUE];
        $options['refresh_interval'] = ['default' => 5];
        $options['date_format'] = ['default' => 'short'];
        $options['show_time_ago'] = ['default' => TRUE];
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function buildOptionsForm(&$form, FormStateInterface $form_state)
    {
        parent::buildOptionsForm($form, $form_state);

        $form['auto_refresh'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Auto-refresh times'),
            '#default_value' => $this->options['auto_refresh'],
            '#description' => $this->t('Periodically re-fetch the import times via AJAX.'),
        ];

        $form['refresh_interval'] = [
            '#type' => 'number',
            '#title' => $this->t('Refresh interval (seconds)'),
            '#min' => $this->minimumRefreshInterval,
            '#default_value' => $this->options['refresh_interval'],
            '#description' => $this->t('How often to refresh the import times when auto-refresh is enabled.'),
            '#states' => [
                'visible' => [
                    ':input[name="options[auto_refresh]"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['date_format'] = [
            '#type' => 'select',
            '#title' => $this->t('Date format'),
            '#options' => [
                'short' => $this->t('Short'),
                'medium' => $this->t('Medium'),
                'long' => $this->t('Long'),
            ],
            '#default_value' => $this->options['date_format'],
        ];

        $form['show_time_ago'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Show relative time (e.g., "2 hours ago")'),
            '#default_value' => $this->options['show_time_ago'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $feed = $this->getEntity($values);

        if (!$feed instanceof FeedInterface) {
            return '';
        }

        $build = $this->buildImportTimes($feed);
        return $this->wrapWithAutoRefreshContainer($build, $feed);
    }

    /**
     * Builds the import times render array.
     */
    protected function buildImportTimes(FeedInterface $feed): array
    {
        $last_import = $feed->getImportedTime();
        $next_import = $feed->getNextImportTime();
        $show_time_ago = $this->options['show_time_ago'] ?? TRUE;
        $date_format = $this->options['date_format'] ?? 'short';

        // Format last import
        $last_import_text = $this->t('Never');
        if ($last_import) {
            if ($show_time_ago) {
                $last_import_text = $this->dateFormatter->formatTimeDiffSince($last_import);
            } else {
                $last_import_text = $this->dateFormatter->format($last_import, $date_format);
            }
        }

        // Format next import
        $next_import_text = $this->t('Not scheduled');
        if ($next_import) {
            $current_time = \Drupal::time()->getRequestTime();
            if ($next_import <= $current_time) {
                $next_import_text = $this->t('Now');
            } else {
                if ($show_time_ago) {
                    $next_import_text = $this->dateFormatter->formatTimeDiffUntil($next_import);
                } else {
                    $next_import_text = $this->dateFormatter->format($next_import, $date_format);
                }
            }
        }

        // Check if feed is currently importing
        $is_importing = FALSE;
        try {
            $is_importing = $feed->isLocked();
        } catch (\Exception $e) {
            // Ignore errors
        }

        return [
            '#type' => 'inline_template',
            '#template' => '<div class="feed-import-times" data-importing="{{ is_importing }}"><div class="last-import"><span class="label">Last:</span> <span class="value">{{ last_import }}</span></div><div class="next-import"><span class="label">Next:</span> <span class="value">{{ next_import }}</span></div></div>',
            '#context' => [
                'last_import' => $last_import_text,
                'next_import' => $next_import_text,
                'is_importing' => $is_importing ? '1' : '0',
            ],
            '#attached' => [
                'library' => [
                    'supplier_products/feed-import-times',
                ],
            ],
        ];
    }
}
