<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to provide human-readable import period from feed type.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("feed_import_period")
 */
class FeedImportPeriod extends FieldPluginBase
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The date formatter service.
     *
     * @var \Drupal\Core\Datetime\DateFormatterInterface
     */
    protected $dateFormatter;

    /**
     * Constructs a FeedImportPeriod object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
     *   The date formatter service.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager,
        DateFormatterInterface $date_formatter
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
        $this->dateFormatter = $date_formatter;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager'),
            $container->get('date.formatter')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $feed = $this->getEntity($values);

        if (!$feed) {
            return '';
        }

        // Get the feed type entity
        $feed_type_id = $feed->bundle();
        try {
            $feed_type = $this->entityTypeManager->getStorage('feeds_feed_type')->load($feed_type_id);

            if (!$feed_type) {
                return '';
            }

            // Get the import period in seconds
            $import_period = $feed_type->getImportPeriod();

            // If set to never (0 or -1)
            if ($import_period <= 0) {
                return $this->t('Never');
            }

            // Format the interval
            return $this->dateFormatter->formatInterval($import_period);
        } catch (\Exception $e) {
            return '';
        }
    }
}
