<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\Core\Url;
use Drupal\feeds\FeedInterface;

/**
 * Shared helpers for auto-refresh capable feed Views fields.
 */
trait FeedAutoRefreshTrait
{

    /**
     * Minimum interval (seconds) between refresh requests.
     */
    protected int $minimumRefreshInterval = 1;

    /**
     * Returns whether auto-refresh is enabled for this field instance.
     */
    protected function isAutoRefreshEnabled(): bool
    {
        return !empty($this->options['auto_refresh']);
    }

    /**
     * Returns the configured refresh interval bounded by the minimum.
     */
    protected function getRefreshInterval(): int
    {
        $interval = (int) ($this->options['refresh_interval'] ?? 5);
        return max($this->minimumRefreshInterval, $interval);
    }

    /**
     * Wraps the rendered field output with metadata for the JS behavior.
     */
    protected function wrapWithAutoRefreshContainer(array $build, FeedInterface $feed): array
    {
        $wrapper = [
            '#type' => 'container',
            '#attributes' => [
                'class' => ['feed-field-wrapper'],
                'data-feed-refresh-enabled' => $this->isAutoRefreshEnabled() ? '1' : '0',
            ],
            'content' => [
                '#type' => 'container',
                '#attributes' => [
                    'class' => ['feed-field-content'],
                    'data-feed-field-content' => '1',
                ],
                'rendered' => $build,
            ],
        ];

        if ($this->isAutoRefreshEnabled() && ($attributes = $this->buildRefreshMetadata($feed))) {
            $wrapper['#attributes'] += $attributes;
            $wrapper['#attached']['library'][] = 'supplier_products/field-refresh';
        }

        return $wrapper;
    }

    /**
     * Builds wrapper attributes describing how to refresh a field.
     */
    protected function buildRefreshMetadata(FeedInterface $feed): array
    {
        $view = $this->view ?? NULL;
        $feed_id = $feed->id();

        if (!$view || !$feed_id || !$view->storage) {
            return [];
        }

        $view_id = $view->storage->id();
        $display_id = $view->current_display ?? 'default';
        $field_id = $this->options['id'] ?? $this->definition['id'] ?? '';

        if (!$view_id || !$display_id || !$field_id) {
            return [];
        }

        $url = Url::fromRoute('supplier_products.field_refresh', [
            'feed' => $feed_id,
            'view_id' => $view_id,
            'display_id' => $display_id,
            'field_id' => $field_id,
        ], ['absolute' => TRUE])->toString();

        return [
            'data-feed-feed-id' => $feed_id,
            'data-feed-view-id' => $view_id,
            'data-feed-display-id' => $display_id,
            'data-feed-field-id' => $field_id,
            'data-feed-refresh-url' => $url,
            'data-feed-refresh-interval' => (string) $this->getRefreshInterval(),
        ];
    }
}
