<?php

namespace Drupal\supplier_products\Plugin\views\field;

use Drupal\commerce_price\PriceFormatterInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Renders the lowest variation price for a commerce product.
 *
 * @ViewsField("commerce_product_variation_price")
 */
class CommerceProductVariationPrice extends FieldPluginBase implements ContainerFactoryPluginInterface
{

    /**
     * Commerce price formatter.
     */
    protected PriceFormatterInterface $priceFormatter;

    /**
     * {@inheritdoc}
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        PriceFormatterInterface $price_formatter
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->priceFormatter = $price_formatter;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('commerce_price.price_formatter')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Avoid altering the SQL query; price is calculated in render().
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $product = $this->getEntity($values);

        if (!$product || !$product->hasField('variations')) {
            return '';
        }

        $lowest_price = NULL;
        foreach ($product->getVariations() as $variation) {
            if (!$variation->hasField('price') || $variation->get('price')->isEmpty()) {
                continue;
            }

            $price = $variation->getPrice();
            if ($price && ($lowest_price === NULL || $price->lessThan($lowest_price))) {
                $lowest_price = $price;
            }
        }

        if ($lowest_price === NULL) {
            return '';
        }

        return $this->priceFormatter->format($lowest_price->getNumber(), $lowest_price->getCurrencyCode());
    }
}
