<?php

declare(strict_types=1);

namespace Drupal\supplier_products\EventSubscriber;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\feeds\Event\EntityEvent;
use Drupal\feeds\Event\FeedsEvents;
use Drupal\feeds\FeedTypeInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Restores original field values for mappings marked as "skip on update".
 */
class SkipOnUpdateSubscriber implements EventSubscriberInterface
{
    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents(): array
    {
        return [
            FeedsEvents::PROCESS_ENTITY_PRESAVE => ['onPresave', 100],
        ];
    }

    /**
     * Restores original field values for fields marked as "skip on update".
     */
    public function onPresave(EntityEvent $event): void
    {
        $feed = $event->getFeed();
        $entity = $event->getEntity();

        // Only apply to existing entities (updates), not new entities.
        if ($entity->isNew()) {
            return;
        }

        // Entity must be a content entity with an ID.
        if (!$entity instanceof ContentEntityInterface || !$entity->id()) {
            return;
        }

        // Load the original entity from storage since $entity->original is not
        // set at PROCESS_ENTITY_PRESAVE time.
        $original = \Drupal::entityTypeManager()
            ->getStorage($entity->getEntityTypeId())
            ->loadUnchanged($entity->id());

        if (!$original instanceof ContentEntityInterface) {
            return;
        }

        $feed_type = $feed->getType();
        if (!$feed_type instanceof FeedTypeInterface) {
            return;
        }

        $mappings = $feed_type->getMappings();

        foreach ($mappings as $mapping) {
            // Check if this mapping is marked to skip on update.
            if (empty($mapping['skip_on_update'])) {
                continue;
            }

            $target = $mapping['target'] ?? '';
            if ($target === '' || $target === 'feeds_item') {
                continue;
            }

            // Restore the original field value.
            if ($entity->hasField($target) && $original->hasField($target)) {
                $original_value = $original->get($target)->getValue();
                $entity->set($target, $original_value);
            }
        }
    }
}
