<?php

namespace Drupal\supplier_products\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\feeds\FeedInterface;
use Drupal\views\Entity\View as ViewEntity;
use Drupal\views\ViewExecutableFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * AJAX controller that returns refreshed feed field markup.
 */
class FeedFieldRefreshController implements ContainerInjectionInterface
{

    /**
     * Allowed Views field plugin IDs for refresh requests.
     */
    protected const SUPPORTED_FIELD_PLUGINS = [
        'feed_progress',
        'feed_import_times',
        'feed_items_imported',
    ];

    /**
     * The entity type manager.
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The Views executable factory.
     */
    protected ViewExecutableFactory $viewExecutableFactory;

    /**
     * The renderer service.
     */
    protected RendererInterface $renderer;

    /**
     * The time service.
     */
    protected TimeInterface $time;

    /**
     * The date formatter service.
     */
    protected DateFormatterInterface $dateFormatter;

    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        ViewExecutableFactory $view_executable_factory,
        RendererInterface $renderer,
        TimeInterface $time,
        DateFormatterInterface $date_formatter,
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->viewExecutableFactory = $view_executable_factory;
        $this->renderer = $renderer;
        $this->time = $time;
        $this->dateFormatter = $date_formatter;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('entity_type.manager'),
            $container->get('views.executable'),
            $container->get('renderer'),
            $container->get('datetime.time'),
            $container->get('date.formatter'),
        );
    }

    /**
     * Returns refreshed field markup for a feed Views field.
     */
    public function refresh(FeedInterface $feed, string $view_id, string $display_id, string $field_id): JsonResponse
    {
        /** @var \Drupal\views\Entity\View|null $view_entity */
        $view_entity = $this->entityTypeManager->getStorage('view')->load($view_id);
        if (!$view_entity instanceof ViewEntity) {
            throw new NotFoundHttpException();
        }

        $view = $this->viewExecutableFactory->get($view_entity);
        if (!$view->setDisplay($display_id) || !$view->display_handler) {
            throw new NotFoundHttpException();
        }

        if (!$view->access($display_id)) {
            throw new AccessDeniedHttpException();
        }

        $field_handler = $view->display_handler->getHandler('field', $field_id);
        if (!$field_handler) {
            throw new NotFoundHttpException();
        }

        $plugin_id = $field_handler->getPluginId();
        if (!in_array($plugin_id, self::SUPPORTED_FIELD_PLUGINS, TRUE)) {
            throw new NotFoundHttpException();
        }

        $options = $field_handler->options ?? [];
        if (empty($options['auto_refresh'])) {
            throw new AccessDeniedHttpException($this->t('Auto-refresh disabled for this field.'));
        }

        // Build the field markup based on plugin type
        $build = match ($plugin_id) {
            'feed_progress' => $this->buildProgressBar($feed),
            'feed_import_times' => $this->buildImportTimes($feed, $options),
            'feed_items_imported' => $this->buildItemsImported($feed),
            default => throw new NotFoundHttpException($this->t('Unsupported field type.')),
        };

        // Ensure refreshed markup is never cached downstream.
        $build['#cache']['max-age'] = 0;
        $html = $this->renderer->renderRoot($build);

        $response = new JsonResponse([
            'status' => 'ok',
            'html' => (string) $html,
            'generated' => $this->time->getRequestTime(),
        ]);
        $response->setMaxAge(0);
        $response->headers->set('Cache-Control', 'no-store, no-cache, must-revalidate');

        return $response;
    }

    /**
     * Builds the progress bar render array.
     */
    protected function buildProgressBar(FeedInterface $feed): array
    {
        // Calculate progress percentage
        $percentage = 0;

        // Check if feed is currently importing
        try {
            // Only show progress if actively importing (locked)
            if ($feed->isLocked()) {
                // Get current import progress from the feed state
                if (method_exists($feed, 'getState')) {
                    $state = $feed->getState(\Drupal\feeds\StateInterface::PROCESS);
                    if ($state) {
                        $total = $state->total ?? 0;

                        // If state total is 0, try using feed item count
                        if ($total == 0) {
                            $total = (int) $feed->getItemCount();
                        }

                        if ($total > 0) {
                            // Calculate progress based on processed items
                            $processed = ($state->created ?? 0) + ($state->updated ?? 0) +
                                ($state->deleted ?? 0) + ($state->skipped ?? 0) +
                                ($state->failed ?? 0);
                            $percentage = min(100, max(0, round(($processed / $total) * 100)));
                        } else {
                            // Fallback to state progress property (0.0 to 1.0)
                            $progress_ratio = $state->progress ?? 0;
                            $percentage = min(100, max(0, round($progress_ratio * 100)));
                        }
                    }
                }
            }
            // If not locked/importing, show 0%
        } catch (\Exception $e) {
            // If there's an error getting state, default to 0%
            $percentage = 0;
        }

        // Build the progress bar HTML
        return [
            '#type' => 'inline_template',
            '#template' => '<div class="feed-progress-wrapper" data-percentage="{{ percentage }}"><div class="percentage-label" data-percentage="{{ percentage }}">{{ percentage }}%</div><div class="progress-bar-container"><div class="progress-bar" style="width: {{ percentage }}%;"></div></div></div>',
            '#context' => [
                'percentage' => $percentage,
            ],
            '#attached' => [
                'library' => [
                    'supplier_products/feed-progress',
                ],
            ],
        ];
    }

    /**
     * Builds the import times render array.
     */
    protected function buildImportTimes(FeedInterface $feed, array $options): array
    {
        $last_import = $feed->getImportedTime();
        $next_import = $feed->getNextImportTime();
        $show_time_ago = $options['show_time_ago'] ?? TRUE;
        $date_format = $options['date_format'] ?? 'short';

        // Format last import
        $last_import_text = $this->t('Never');
        if ($last_import) {
            if ($show_time_ago) {
                $last_import_text = $this->dateFormatter->formatTimeDiffSince($last_import);
            } else {
                $last_import_text = $this->dateFormatter->format($last_import, $date_format);
            }
        }

        // Format next import
        $next_import_text = $this->t('Not scheduled');
        if ($next_import) {
            $current_time = $this->time->getRequestTime();
            if ($next_import <= $current_time) {
                $next_import_text = $this->t('Now');
            } else {
                if ($show_time_ago) {
                    $next_import_text = $this->dateFormatter->formatTimeDiffUntil($next_import);
                } else {
                    $next_import_text = $this->dateFormatter->format($next_import, $date_format);
                }
            }
        }

        // Check if feed is currently importing
        $is_importing = FALSE;
        try {
            $is_importing = $feed->isLocked();
        } catch (\Exception $e) {
            // Ignore errors
        }

        return [
            '#type' => 'inline_template',
            '#template' => '<div class="feed-import-times" data-importing="{{ is_importing }}"><div class="last-import"><span class="label">Last:</span> <span class="value">{{ last_import }}</span></div><div class="next-import"><span class="label">Next:</span> <span class="value">{{ next_import }}</span></div></div>',
            '#context' => [
                'last_import' => $last_import_text,
                'next_import' => $next_import_text,
                'is_importing' => $is_importing ? '1' : '0',
            ],
            '#attached' => [
                'library' => [
                    'supplier_products/feed-import-times',
                ],
            ],
        ];
    }

    /**
     * Builds the items imported render array.
     */
    protected function buildItemsImported(FeedInterface $feed): array
    {
        $count = (int) $feed->getItemCount();

        return [
            '#markup' => $count,
        ];
    }

    /**
     * Translation wrapper.
     */
    protected function t(string $string): string
    {
        return \Drupal::translation()->translate($string);
    }
}
