(function (Drupal, once) {
  "use strict";

  const timers = new WeakMap();

  function stopTimer(element) {
    const timerId = timers.get(element);
    if (timerId) {
      window.clearInterval(timerId);
      timers.delete(element);
    }
  }

  function replaceContent(element, html) {
    const content = element.querySelector("[data-feed-field-content]");
    if (content) {
      content.innerHTML = html;
    }
  }

  function scheduleRefresh(element) {
    const intervalSeconds = parseInt(element.dataset.feedRefreshInterval, 10);
    const refreshUrl = element.dataset.feedRefreshUrl;

    if (!refreshUrl || Number.isNaN(intervalSeconds) || intervalSeconds <= 0) {
      return;
    }

    stopTimer(element);

    const tick = () => {
      fetch(refreshUrl, { credentials: "same-origin" })
        .then((response) => {
          if (!response.ok) {
            throw new Error("Request failed");
          }
          return response.json();
        })
        .then((data) => {
          if (!data || data.status !== "ok" || typeof data.html !== "string") {
            throw new Error("Invalid payload");
          }
          replaceContent(element, data.html);
          element.classList.remove("feed-refresh-error");
        })
        .catch(() => {
          element.classList.add("feed-refresh-error");
        });
    };

    // Fetch immediately so the UI stays fresh, then schedule.
    tick();
    const timerId = window.setInterval(tick, intervalSeconds * 1000);
    timers.set(element, timerId);
  }

  Drupal.behaviors.feedFieldRefresh = {
    attach(context) {
      once(
        "feed-field-refresh",
        '.feed-field-wrapper[data-feed-refresh-enabled="1"]',
        context
      ).forEach((element) => {
        scheduleRefresh(element);
      });
    },
    detach(context, settings, trigger) {
      if (trigger !== "unload") {
        return;
      }
      const elements = context.querySelectorAll
        ? context.querySelectorAll(
            '.feed-field-wrapper[data-feed-refresh-enabled="1"]'
          )
        : [];
      elements.forEach((element) => stopTimer(element));
    },
  };
})(Drupal, once);
