<?php

namespace Drupal\stockmount_xml\Plugin\views\field;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to display product images with configurable options.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("stockmount_product_images")
 */
class ProductImages extends FieldPluginBase
{

    /**
     * The entity field manager.
     *
     * @var \Drupal\Core\Entity\EntityFieldManagerInterface
     */
    protected $entityFieldManager;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The file URL generator.
     *
     * @var \Drupal\Core\File\FileUrlGeneratorInterface
     */
    protected $fileUrlGenerator;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
        $instance->entityFieldManager = $container->get('entity_field.manager');
        $instance->entityTypeManager = $container->get('entity_type.manager');
        $instance->fileUrlGenerator = $container->get('file_url_generator');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    protected function defineOptions()
    {
        $options = parent::defineOptions();
        $options['image_style'] = ['default' => ''];
        $options['image_field'] = ['default' => 'field_product_images'];
        $options['media_image_field'] = ['default' => 'field_media_image'];
        $options['wrap_tag'] = ['default' => ''];
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function buildOptionsForm(&$form, FormStateInterface $form_state)
    {
        parent::buildOptionsForm($form, $form_state);

        // Get available image styles.
        $image_styles = image_style_options(FALSE);

        $form['image_style'] = [
            '#type' => 'select',
            '#title' => $this->t('Image style'),
            '#description' => $this->t('Select the image style to apply to images.'),
            '#options' => ['' => $this->t('- Original -')] + $image_styles,
            '#default_value' => $this->options['image_style'],
        ];

        // Get media reference fields from commerce_product.
        $product_fields = $this->getEntityReferenceFields('commerce_product', 'media');
        $form['image_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Product image field'),
            '#description' => $this->t('Select the media reference field on the product.'),
            '#options' => $product_fields,
            '#default_value' => $this->options['image_field'],
            '#required' => TRUE,
        ];

        // Get image fields from media entities.
        $media_fields = $this->getImageFieldsFromMedia();
        $form['media_image_field'] = [
            '#type' => 'select',
            '#title' => $this->t('Media image field'),
            '#description' => $this->t('Select the image field on the media entity.'),
            '#options' => $media_fields,
            '#default_value' => $this->options['media_image_field'],
            '#required' => TRUE,
        ];

        $form['wrap_tag'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Wrap each URL in tag'),
            '#description' => $this->t('XML tag to wrap each image URL (e.g., "Resim" outputs &lt;Resim&gt;url&lt;/Resim&gt;). Leave empty for no wrapping.'),
            '#default_value' => $this->options['wrap_tag'],
            '#size' => 20,
        ];
    }

    /**
     * Get entity reference fields targeting a specific entity type.
     */
    protected function getEntityReferenceFields($entity_type, $target_type)
    {
        $options = [];
        $field_definitions = $this->entityFieldManager
            ->getFieldStorageDefinitions($entity_type);

        foreach ($field_definitions as $field_name => $definition) {
            if ($definition->getType() === 'entity_reference') {
                $settings = $definition->getSettings();
                if (!empty($settings['target_type']) && $settings['target_type'] === $target_type) {
                    $options[$field_name] = $field_name;
                }
            }
        }

        return $options ?: ['' => $this->t('- No fields found -')];
    }

    /**
     * Get image fields from media entities.
     */
    protected function getImageFieldsFromMedia()
    {
        $options = [];
        $field_definitions = $this->entityFieldManager
            ->getFieldStorageDefinitions('media');

        foreach ($field_definitions as $field_name => $definition) {
            if ($definition->getType() === 'image') {
                $options[$field_name] = $field_name;
            }
        }

        return $options ?: ['' => $this->t('- No fields found -')];
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // This field doesn't need a query - data comes from entity.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $entity = $values->_entity;

        if (!$entity) {
            return '';
        }

        // Handle variations - get parent product.
        $product = NULL;
        if (method_exists($entity, 'getProduct')) {
            $product = $entity->getProduct();
        } elseif ($entity->getEntityTypeId() === 'commerce_product') {
            $product = $entity;
        }

        if (!$product) {
            return '';
        }

        $image_field = $this->options['image_field'];
        $media_image_field = $this->options['media_image_field'];
        $wrap_tag = $this->options['wrap_tag'];

        if (!$product->hasField($image_field) || $product->get($image_field)->isEmpty()) {
            return '';
        }

        // Load image style if configured.
        $image_style = NULL;
        if (!empty($this->options['image_style'])) {
            $image_style = $this->entityTypeManager
                ->getStorage('image_style')
                ->load($this->options['image_style']);
        }

        $image_urls = [];

        foreach ($product->get($image_field) as $item) {
            if (!$item->entity) {
                continue;
            }

            $media = $item->entity;
            if (!$media->hasField($media_image_field) || $media->get($media_image_field)->isEmpty()) {
                continue;
            }

            $file = $media->get($media_image_field)->entity;
            if (!$file) {
                continue;
            }

            $uri = $file->getFileUri();
            if ($image_style) {
                $image_urls[] = $image_style->buildUrl($uri);
            } else {
                $image_urls[] = $this->fileUrlGenerator->generateAbsoluteString($uri);
            }
        }

        if (empty($image_urls)) {
            return '';
        }

        // Always return string - row_alter hook will handle wrap_tag transformation.
        return implode('|||', $image_urls);
    }
}
