<?php

namespace Drupal\stockmount_xml\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;

/**
 * Field handler to display EAN-13 barcode from product ID.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("stockmount_barkod")
 */
class Barkod extends FieldPluginBase
{

    /**
     * {@inheritdoc}
     */
    protected function defineOptions()
    {
        $options = parent::defineOptions();
        $options['prefix'] = ['default' => '86900'];
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function buildOptionsForm(&$form, FormStateInterface $form_state)
    {
        parent::buildOptionsForm($form, $form_state);

        $form['prefix'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Barcode prefix'),
            '#description' => $this->t('The prefix for the EAN-13 barcode (e.g., 86900 for Turkey). Must be 5 digits.'),
            '#default_value' => $this->options['prefix'],
            '#size' => 10,
            '#maxlength' => 5,
            '#required' => TRUE,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // This field doesn't need a query.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $entity = $values->_entity;

        if (!$entity || !method_exists($entity, 'getProduct')) {
            return '';
        }

        $product = $entity->getProduct();
        if (!$product) {
            return '';
        }

        return $this->generateEan13($product->id());
    }

    /**
     * Generates an EAN-13 barcode from a product ID.
     *
     * @param int $product_id
     *   The product ID.
     *
     * @return string
     *   A valid 13-digit EAN-13 barcode.
     */
    protected function generateEan13($product_id)
    {
        // Use configured prefix.
        $prefix = $this->options['prefix'] ?: '86900';
        $padded_id = str_pad($product_id, 7, '0', STR_PAD_LEFT);
        $code = substr($prefix . $padded_id, 0, 12);

        // Calculate check digit.
        $sum = 0;
        for ($i = 0; $i < 12; $i++) {
            $digit = (int) $code[$i];
            $sum += ($i % 2 === 0) ? $digit : $digit * 3;
        }
        $check_digit = (10 - ($sum % 10)) % 10;

        return $code . $check_digit;
    }
}
