<?php

namespace Drupal\sogan_product_merger\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\node\NodeInterface;
use Drupal\sogan_commerce_product\Service\ProductBuilder;

/**
 * Manages product merging operations via file equivalence maps.
 */
class ProductMergerManager
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The product builder service.
     *
     * @var \Drupal\sogan_commerce_product\Service\ProductBuilder
     */
    protected $productBuilder;

    /**
     * The logger channel.
     *
     * @var \Drupal\Core\Logger\LoggerChannelInterface
     */
    protected $logger;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * The queue worker manager.
     *
     * @var \Drupal\Component\Plugin\PluginManagerInterface|null
     */
    protected $queueWorkerManager;

    /**
     * Constructs a new ProductMergerManager object.
     *
     * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
     *   The entity type manager.
     * @param \Drupal\sogan_commerce_product\Service\ProductBuilder $product_builder
     *   The product builder service.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param mixed $queue_worker_manager
     *   The queue worker manager (optional).
     */
    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        ProductBuilder $product_builder,
        LoggerChannelFactoryInterface $logger_factory,
        ConfigFactoryInterface $config_factory,
        $queue_worker_manager = NULL
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->productBuilder = $product_builder;
        $this->logger = $logger_factory->get('sogan_product_merger');
        $this->configFactory = $config_factory;
        $this->queueWorkerManager = $queue_worker_manager;
    }

    /**
     * Exports product source mappings as flat groups of supplier:sku pairs.
     *
     * Each group represents one product's supplier sources.
     * Format: [[{"supplier": "Name", "sku": "ABC"}, ...], ...]
     *
     * @param string $destination
     *   The destination file path (e.g., /tmp/map.json).
     *
     * @return int
     *   The number of groups exported.
     */
    public function exportEquivalenceMap(string $destination): int
    {
        $product_ids = $this->entityTypeManager->getStorage('commerce_product')->getQuery()
            ->accessCheck(FALSE)
            ->execute();

        $map = [];

        if (!empty($product_ids)) {
            foreach (array_chunk($product_ids, 50) as $chunk_ids) {
                $products = $this->entityTypeManager->getStorage('commerce_product')->loadMultiple($chunk_ids);

                foreach ($products as $product) {
                    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
                    $variations = $product->getVariations();

                    if (empty($variations)) {
                        continue;
                    }

                    // Collect all supplier:sku pairs for this product
                    $sp_list = [];

                    foreach ($variations as $variation) {
                        /** @var \Drupal\commerce_product\Entity\ProductVariationInterface $variation */
                        if (!$variation->hasField('field_source_supplier_products') || $variation->get('field_source_supplier_products')->isEmpty()) {
                            continue;
                        }

                        $supplier_products = $variation->get('field_source_supplier_products')->referencedEntities();

                        foreach ($supplier_products as $node) {
                            /** @var \Drupal\node\NodeInterface $node */
                            $supplier_name = '';
                            if ($node->hasField('field_supplier') && !$node->get('field_supplier')->isEmpty()) {
                                $term = $node->get('field_supplier')->entity;
                                if ($term) {
                                    $supplier_name = $term->label();
                                }
                            }

                            $sku = '';
                            if ($node->hasField('field_supplier_sku') && !$node->get('field_supplier_sku')->isEmpty()) {
                                $sku = $node->get('field_supplier_sku')->value;
                            }

                            if ($supplier_name && $sku) {
                                $sp_list[] = [
                                    'supplier' => $supplier_name,
                                    'sku' => $sku,
                                ];
                            }
                        }
                    }

                    if (!empty($sp_list)) {
                        // Sort for consistent output
                        usort($sp_list, function ($a, $b) {
                            $cmp = strcmp($a['supplier'], $b['supplier']);
                            return $cmp !== 0 ? $cmp : strcmp($a['sku'], $b['sku']);
                        });

                        $map[] = $sp_list;
                    }
                }
            }
        }

        $json = json_encode($map, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        file_put_contents($destination, $json);

        $this->logger->info('Exported product map with @count groups to @dest', [
            '@count' => count($map),
            '@dest' => $destination,
        ]);

        return count($map);
    }

    /**
     * Creates products from an equivalence map file.
     *
     * Expected format: [[{"supplier": "Name", "sku": "ABC"}, ...], ...]
     * Each inner array = one product group to merge.
     *
     * @param string $source
     *   The source file path.
     *
     * @return int
     *   The number of products touched (created or updated).
     */
    public function createProductsFromMap(string $source): int
    {
        if (!file_exists($source)) {
            throw new \InvalidArgumentException("Source file not found: $source");
        }

        $json = file_get_contents($source);
        $map = json_decode($json, TRUE);

        if (!is_array($map)) {
            throw new \UnexpectedValueException("Invalid JSON in source file.");
        }

        // Validate format: must be array of arrays of {supplier, sku} objects
        if (!empty($map) && isset($map[0])) {
            $first = $map[0];
            if (!is_array($first) || !isset($first[0]) || !isset($first[0]['supplier']) || !isset($first[0]['sku'])) {
                throw new \UnexpectedValueException(
                    "Invalid format. Expected: [[{\"supplier\": \"X\", \"sku\": \"Y\"}, ...], ...]"
                );
            }
        }

        $touched_count = 0;

        foreach ($map as $group) {
            if (!is_array($group) || empty($group)) {
                continue;
            }

            $result = $this->processGroup($group);
            if ($result) {
                $touched_count++;
            }
        }

        return $touched_count;
    }

    /**
     * Process a flat group of supplier/sku pairs.
     *
     * Acts exactly like "Merge & Create Commerce Product" VBO action.
     *
     * @param array $group
     *   Array of items with 'supplier' and 'sku'.
     *
     * @return bool
     *   TRUE if product was created/updated.
     */
    protected function processGroup(array $group): bool
    {
        $group_label = implode(' + ', array_map(function ($i) {
            return ($i['supplier'] ?? '?') . ':' . ($i['sku'] ?? '?');
        }, $group));

        // Resolve supplier nodes from supplier name + SKU pairs
        $valid_nodes = [];
        foreach ($group as $item) {
            $node = $this->resolveSupplierNode($item);
            if ($node) {
                $unique_key = ($item['supplier'] ?? '') . '_' . ($item['sku'] ?? '');
                $valid_nodes[$unique_key] = $node;
            }
        }

        if (empty($valid_nodes)) {
            $this->logger->warning('No valid supplier nodes found for group: @label', ['@label' => $group_label]);
            return FALSE;
        }

        // Check if any node is already linked to a product
        foreach ($valid_nodes as $node) {
            $query = $this->entityTypeManager->getStorage('commerce_product_variation')->getQuery()
                ->accessCheck(FALSE)
                ->condition('field_source_supplier_products', $node->id());
            $vids = $query->execute();

            if (!empty($vids)) {
                $this->logger->info('Nodes in group @label already linked, skipping.', ['@label' => $group_label]);
                return FALSE;
            }
        }

        // Run AI enhancement if enabled (same as MergeCreateCommerceProduct action)
        $valid_nodes = $this->runAiEnhancementIfEnabled(array_values($valid_nodes));

        // Create new product using the exact same method as VBO action
        try {
            $product = $this->productBuilder->createProductFromSuppliers($valid_nodes);
            if ($product) {
                $this->logger->info('Created product @title with @count supplier nodes.', [
                    '@title' => $product->label(),
                    '@count' => count($valid_nodes),
                ]);
                return TRUE;
            }
        } catch (\Exception $e) {
            $this->logger->error('Failed to create product for group @label: @error', [
                '@label' => $group_label,
                '@error' => $e->getMessage(),
            ]);
        }

        return FALSE;
    }

    /**
     * Runs AI enhancement on entities if enabled and enhance_on_merge is set.
     *
     * Mirrors the logic from MergeCreateCommerceProduct::runAiEnhancementWithMerge().
     *
     * @param array $entities
     *   The supplier product nodes.
     *
     * @return array
     *   The (possibly reloaded) entities after AI enhancement.
     */
    protected function runAiEnhancementIfEnabled(array $entities): array
    {
        if (!$this->queueWorkerManager) {
            return $entities;
        }

        $aiConfig = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $aiEnabled = (bool) $aiConfig->get('enabled');
        $enhanceOnMerge = (bool) $aiConfig->get('enhance_on_merge');

        if (!$aiEnabled || !$enhanceOnMerge) {
            return $entities;
        }

        $editableConfig = $this->configFactory->getEditable('supplier_products_ai_rewrite.settings');

        // Store original context instructions.
        $originalContext = $editableConfig->get('context_instructions') ?? '';
        $mergeInstructions = $editableConfig->get('merge_instructions') ?? '';

        try {
            // Temporarily append merge instructions to context.
            if (!empty($mergeInstructions)) {
                $combinedContext = $originalContext . "\n\n" . $mergeInstructions;
                $editableConfig->set('context_instructions', $combinedContext)->save();
            }

            /** @var \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer $queueWorker */
            $queueWorker = $this->queueWorkerManager->createInstance('ai_content_enhancer');

            // Process all entities as a batch.
            foreach ($entities as $entity) {
                if (!$entity instanceof NodeInterface) {
                    continue;
                }

                $item = [
                    'nid' => $entity->id(),
                    'tasks' => ['title', 'description', 'attributes', 'categorize', 'brand'],
                ];

                $queueWorker->processItem($item);
            }

            // Flush batch to ensure all items are processed together.
            if (class_exists('\Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer')) {
                \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer::flushBatch();
            }

            // Reload all entities to get AI-enhanced fields.
            $reloadedEntities = [];
            $nodeStorage = $this->entityTypeManager->getStorage('node');
            foreach ($entities as $entity) {
                if ($entity instanceof NodeInterface) {
                    $reloadedEntities[] = $nodeStorage->load($entity->id());
                }
            }

            $this->logger->notice('AI enhancement completed for @count entities in merge mode', [
                '@count' => count($reloadedEntities),
            ]);

            return $reloadedEntities;
        } catch (\Exception $e) {
            $this->logger->warning('AI enhancement failed during merge: @message', [
                '@message' => $e->getMessage(),
            ]);
            // Return original entities if AI enhancement fails.
            return $entities;
        } finally {
            // Restore original context instructions.
            if (!empty($mergeInstructions)) {
                $editableConfig->set('context_instructions', $originalContext)->save();
            }
        }
    }

    /**
     * Resolve a supplier node from supplier name and SKU.
     *
     * @param array $item
     *   Item with 'supplier' and 'sku'.
     *
     * @return \Drupal\node\NodeInterface|null
     *   The supplier node or NULL.
     */
    protected function resolveSupplierNode(array $item): ?NodeInterface
    {
        $supplier_name = $item['supplier'] ?? NULL;
        $sku = $item['sku'] ?? NULL;

        if (!$supplier_name || !$sku) {
            return NULL;
        }

        $terms = $this->entityTypeManager->getStorage('taxonomy_term')
            ->loadByProperties(['vid' => 'suppliers', 'name' => $supplier_name]);

        if (!$terms) {
            return NULL;
        }

        $term_id = reset($terms)->id();

        $nodes = $this->entityTypeManager->getStorage('node')
            ->loadByProperties([
                'type' => 'supplier_product',
                'field_supplier' => $term_id,
                'field_supplier_sku' => $sku,
            ]);

        return $nodes ? reset($nodes) : NULL;
    }
}
