<?php

namespace Drupal\sogan_product_merger\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sogan_product_merger\Service\ProductMergerManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Provides a form for exporting and importing product equivalence maps.
 */
class ProductMergerUiForm extends FormBase
{

    /**
     * The product merger manager.
     *
     * @var \Drupal\sogan_product_merger\Service\ProductMergerManager
     */
    protected $productMergerManager;

    /**
     * Constructs a new ProductMergerUiForm.
     *
     * @param \Drupal\sogan_product_merger\Service\ProductMergerManager $product_merger_manager
     *   The product merger manager.
     */
    public function __construct(ProductMergerManager $product_merger_manager)
    {
        $this->productMergerManager = $product_merger_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('sogan_product_merger.manager')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'sogan_product_merger_ui_form';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $form['export_details'] = [
            '#type' => 'details',
            '#title' => $this->t('Export Equivalence Map'),
            '#open' => TRUE,
            '#description' => $this->t('Export the current configuration of merged supplier products. This file can be used to import merges on another site.'),
        ];

        $form['export_details']['export'] = [
            '#type' => 'submit',
            '#value' => $this->t('Export Equivalence Map'),
            '#submit' => ['::submitExport'],
        ];

        $form['import_details'] = [
            '#type' => 'details',
            '#title' => $this->t('Import Equivalence Map'),
            '#open' => TRUE,
            '#description' => $this->t('Upload a JSON file containing an equivalence map to merge supplier products.'),
        ];

        $form['import_details']['file'] = [
            '#type' => 'managed_file',
            '#title' => $this->t('Equivalence Map (JSON)'),
            '#upload_validators' => [
                'FileExtension' => ['extensions' => 'json'],
            ],
            '#upload_location' => 'temporary://',
        ];

        $form['import_details']['import'] = [
            '#type' => 'submit',
            '#value' => $this->t('Import and Merge'),
            '#validate' => ['::validateImport'],
            '#submit' => ['::submitImport'],
        ];

        return $form;
    }

    /**
     * Submission handler for the export button.
     */
    public function submitExport(array &$form, FormStateInterface $form_state)
    {
        $filename = 'product_equivalence_map_' . date('Y-m-d_H-i-s') . '.json';
        $uri = 'temporary://' . $filename;

        try {
            $count = $this->productMergerManager->exportEquivalenceMap($uri);

            if ($count > 0) {
                // Return the file download directly.
                // We use EnforcedResponseException to bypass normal form redirect.
                $response = new BinaryFileResponse($uri);
                $response->setContentDisposition('attachment', $filename);
                $response->deleteFileAfterSend(TRUE);

                throw new \Drupal\Core\Form\EnforcedResponseException($response);
            } else {
                $this->messenger()->addWarning($this->t('No equivalence groups found to export.'));
            }
        } catch (\Drupal\Core\Form\EnforcedResponseException $e) {
            // Rethrow so Drupal handles it.
            throw $e;
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Export failed: @error', ['@error' => $e->getMessage()]));
        }
    }

    /**
     * Validation handler for import.
     */
    public function validateImport(array &$form, FormStateInterface $form_state)
    {
        $fid = $form_state->getValue(['file', 0]);
        if (empty($fid)) {
            $form_state->setErrorByName('file', $this->t('Please upload a file.'));
        }
    }

    /**
     * Submission handler for the import button.
     */
    public function submitImport(array &$form, FormStateInterface $form_state)
    {
        $fid = $form_state->getValue(['file', 0]);
        if (!empty($fid)) {
            $file = \Drupal\file\Entity\File::load($fid);
            if ($file) {
                // Set status permanent
                $file->setPermanent();
                $file->save();

                $real_path = \Drupal::service('file_system')->realpath($file->getFileUri());

                try {
                    $count = $this->productMergerManager->createProductsFromMap($real_path);
                    $this->messenger()->addStatus($this->t('Successfully processed @count items from the map.', ['@count' => $count]));
                } catch (\Exception $e) {
                    $this->messenger()->addError($this->t('Import failed: @error', ['@error' => $e->getMessage()]));
                }
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        // Default submit handler if needed, but we use per-button handlers.
    }
}
