<?php

namespace Drupal\sogan_product_merger\Commands;

use Drush\Commands\DrushCommands;
use Drupal\sogan_product_merger\Service\ProductMergerManager;

/**
 * Drush commands for the Sogan Product Merger module.
 */
class ProductMergerCommands extends DrushCommands
{

    /**
     * The product merger manager.
     *
     * @var \Drupal\sogan_product_merger\Service\ProductMergerManager
     */
    protected $productMergerManager;

    /**
     * Constructs a new ProductMergerCommands object.
     *
     * @param \Drupal\sogan_product_merger\Service\ProductMergerManager $product_merger_manager
     *   The product merger manager.
     */
    public function __construct(ProductMergerManager $product_merger_manager)
    {
        parent::__construct();
        $this->productMergerManager = $product_merger_manager;
    }

    /**
     * Exports an equivalence map of supplier products.
     *
     * @param string $path
     *   The destination path for the JSON map file.
     *
     * @command product-merger:export
     * @aliases pme
     * @usage product-merger:export /tmp/map.json
     *   Exports the current product equivalence map to /tmp/map.json.
     */
    public function export($path)
    {
        $this->logger()->notice(dt('Exporting product equivalence map to @path...', ['@path' => $path]));

        try {
            $count = $this->productMergerManager->exportEquivalenceMap($path);
            $this->logger()->success(dt('Successfully exported @count product groups.', ['@count' => $count]));
        } catch (\Exception $e) {
            $this->logger()->error($e->getMessage());
        }
    }

    /**
     * Imports and executes product merges from an equivalence map.
     *
     * @param string $path
     *   The source path of the JSON map file.
     *
     * @command product-merger:import
     * @aliases pmi
     * @usage product-merger:import /tmp/map.json
     *   Reads /tmp/map.json and merges supplier products defined therein.
     */
    public function import($path)
    {
        $this->logger()->notice(dt('Importing product merges from @path...', ['@path' => $path]));

        try {
            $count = $this->productMergerManager->createProductsFromMap($path);
            $this->logger()->success(dt('Successfully merged/created @count products.', ['@count' => $count]));
        } catch (\Exception $e) {
            $this->logger()->error($e->getMessage());
        }
    }
}
