<?php

namespace Drupal\Tests\sogan_commerce_product\Unit\Service;

use Drupal\sogan_commerce_product\Service\PriceCalculator;
use Drupal\sogan_commerce_product\Service\StockManager;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\node\NodeInterface;
use Drupal\Core\Field\FieldItemListInterface;

/**
 * Tests for PriceCalculator service.
 *
 * @group sogan_commerce_product
 * @coversDefaultClass \Drupal\sogan_commerce_product\Service\PriceCalculator
 */
class PriceCalculatorTest extends UnitTestCase
{
    /**
     * @var \Drupal\sogan_commerce_product\Service\PriceCalculator
     */
    protected $priceCalculator;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject|\Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject|\Drupal\sogan_commerce_product\Service\StockManager
     */
    protected $stockManager;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
        $this->stockManager = $this->createMock(StockManager::class);
    }

    /**
     * Helper to create a PriceCalculator with specific config.
     */
    protected function createCalculator(array $merge_policy = [])
    {
        $config = $this->createMock(ImmutableConfig::class);
        $config->method('get')->with('merge_policy')->willReturn($merge_policy);

        $this->configFactory->method('get')->with('sogan_commerce_product.settings')->willReturn($config);

        return new PriceCalculator($this->configFactory, $this->stockManager);
    }

    /**
     * Helper to create a mock supplier node.
     */
    protected function createMockNode($cost, $stock = 10)
    {
        $node = $this->createMock(NodeInterface::class);

        // Mock getSupplierPriceData dependencies
        $node->method('hasField')->willReturnMap([
            ['field_stock', TRUE],
            ['field_cost_price', TRUE],
            ['field_list_price', TRUE],
            ['field_suggested_price', TRUE],
        ]);

        $createField = function ($value) {
            $field = $this->createMock(FieldItemListInterface::class);
            $field->isEmpty()->willReturn(FALSE);
            $field->value = $value;
            $field->number = $value;
            return $field;
        };

        $node->method('get')->willReturnMap([
            ['field_stock', $createField($stock)],
            ['field_cost_price', $createField($cost)],
            ['field_list_price', $createField($cost)], // Using cost as placeholder for list
            ['field_suggested_price', $createField($cost)], // Using cost as placeholder for suggested
        ]);

        return $node;
    }

    /**
     * Tests calculation with Minimum strategy.
     */
    public function testMinimumStrategy()
    {
        $calculator = $this->createCalculator(['cost_price_strategy' => 'minimum']);

        $nodes = [
            $this->createMockNode(100),
            $this->createMockNode(50),
            $this->createMockNode(200),
        ];

        $result = $calculator->calculateCostPrice($nodes);
        $this->assertEquals(50.0, $result);
    }

    /**
     * Tests calculation with Maximum strategy.
     */
    public function testMaximumStrategy()
    {
        $calculator = $this->createCalculator(['cost_price_strategy' => 'maximum']);

        $nodes = [
            $this->createMockNode(100),
            $this->createMockNode(50),
            $this->createMockNode(200),
        ];

        $result = $calculator->calculateCostPrice($nodes);
        $this->assertEquals(200.0, $result);
    }

    /**
     * Tests calculation with Average strategy.
     */
    public function testAverageStrategy()
    {
        $calculator = $this->createCalculator(['cost_price_strategy' => 'average']);

        $nodes = [
            $this->createMockNode(100), // 100
            $this->createMockNode(50),  // 50
            $this->createMockNode(150), // 150
        ];
        // Sum = 300, Count = 3, Avg = 100
        $result = $calculator->calculateCostPrice($nodes);
        $this->assertEquals(100.0, $result);
    }

    /**
     * Tests that zero/negative prices are ignored.
     */
    public function testIgnoresZeroPrices()
    {
        $calculator = $this->createCalculator(['cost_price_strategy' => 'minimum']);

        $nodes = [
            $this->createMockNode(100),
            $this->createMockNode(0),   // Should be ignored
            $this->createMockNode(-50), // Should be ignored
        ];

        $result = $calculator->calculateCostPrice($nodes);
        $this->assertEquals(100.0, $result);
    }

    /**
     * Tests fallback when all prices are invalid.
     */
    public function testAllInvalidPricesReturnsNull()
    {
        $calculator = $this->createCalculator(['cost_price_strategy' => 'minimum']);

        $nodes = [
            $this->createMockNode(0),
            $this->createMockNode(-10),
        ];

        $result = $calculator->calculateCostPrice($nodes);
        $this->assertNull($result);
    }
}
