<?php

namespace Drupal\Tests\sogan_commerce_product\Unit\Service;

use Drupal\Tests\UnitTestCase;
use Drupal\sogan_commerce_product\Service\ContentResolver;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\TermInterface;
use Drupal\Core\Field\FieldItemListInterface;

use Drupal\Core\Field\FieldItemList;

/**
 * @coversDefaultClass \Drupal\sogan_commerce_product\Service\ContentResolver
 * @group sogan_commerce_product
 */
class ContentResolverTest extends UnitTestCase
{

    /**
     * @var \Drupal\sogan_commerce_product\Service\ContentResolver
     */
    protected $contentResolver;

    /**
     * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
     */
    protected $configFactory;

    protected function setUp(): void
    {
        parent::setUp();

        $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
        $config = $this->createMock(ImmutableConfig::class);

        // Mock config with 'weight' strategy
        $config->method('get')->willReturnMap([
            ['merge_policy', ['test_strategy' => 'weight']],
        ]);

        $this->configFactory->method('get')->willReturn($config);

        $this->contentResolver = new ContentResolver($this->configFactory);
    }

    /**
     * Helper to create a mock node with a supplier weight.
     */
    protected function createMockNode($id, $title, $created, $supplierWeight)
    {
        $node = $this->createMock(NodeInterface::class);
        $node->method('id')->willReturn($id);
        $node->method('getCreatedTime')->willReturn($created);

        // helper to create a field list mock with a specific 'value'
        $createStringField = function ($val) {
            $mock = $this->getMockBuilder(FieldItemList::class)
                ->disableOriginalConstructor()
                ->onlyMethods(['__get', 'isEmpty']) // Explicitly mock __get
                ->getMock();

            $mock->method('isEmpty')->willReturn(empty($val));
            $mock->method('__get')->with('value')->willReturn($val);
            return $mock;
        };

        $aiTitleList = $createStringField(NULL);
        $origTitleList = $createStringField($title);

        // Mock 'field_supplier'
        $supplierList = $this->getMockBuilder(FieldItemList::class)
            ->disableOriginalConstructor()
            ->onlyMethods(['__get', 'isEmpty'])
            ->getMock();
        $supplierList->method('isEmpty')->willReturn(false);

        // Create term with field_supplier_weight
        $term = $this->createMock(TermInterface::class);
        // $term->method('getWeight')->willReturn($supplierWeight); // No longer used

        // Create mocked field list for weight
        $weightList = $createStringField($supplierWeight);

        $term->method('hasField')->with('field_supplier_weight')->willReturn(true);
        $term->method('get')->with('field_supplier_weight')->willReturn($weightList);

        $supplierList->method('__get')->with('entity')->willReturn($term);

        $node->method('get')->willReturnMap([
            ['field_ai_title', $aiTitleList],
            ['field_original_product_title', $origTitleList],
            ['field_supplier', $supplierList],
        ]);

        $node->method('hasField')->willReturnMap([
            ['field_ai_title', true],
            ['field_original_product_title', true],
            ['field_supplier', true],
        ]);

        return $node;
    }

    /**
     * Tests standard weight sorting.
     */
    public function testSelectTextByStrategyWeight()
    {
        // Node 1: Weight 10 (Lower priority), Created later
        $node1 = $this->createMockNode(1, 'Title Low Priority', 2000, 10);

        // Node 2: Weight 0 (Higher priority), Created earlier
        $node2 = $this->createMockNode(2, 'Title High Priority', 1000, 0);

        $nodes = [$node1, $node2];

        // Should return Node 2's title because weight 0 < 10
        $result = $this->contentResolver->selectTextByStrategy($nodes, 'title', 'test_strategy');
        $this->assertEquals('Title High Priority', $result);
    }

    /**
     * Tests sorting with equal weights (Tie-breaker: newest created).
     */
    public function testSelectTextByStrategyWeightTie()
    {
        // Node 1: Weight 5, Created 1000 (Older)
        $node1 = $this->createMockNode(1, 'Title Older', 1000, 5);

        // Node 2: Weight 5, Created 2000 (Newer)
        $node2 = $this->createMockNode(2, 'Title Newer', 2000, 5);

        $nodes = [$node1, $node2];

        // Should return Node 2's title because weights equal, but Node 2 is newer
        $result = $this->contentResolver->selectTextByStrategy($nodes, 'title', 'test_strategy');
        $this->assertEquals('Title Newer', $result);
    }
}
