<?php

namespace Drupal\sogan_commerce_product\Tests\Unit;

use Drupal\sogan_commerce_product\Service\ProductBuilder;
use PHPUnit\Framework\TestCase;

class ProductBuilderComputeTest extends TestCase
{
    public function createProductBuilderWithMocks()
    {
        $entity_type_manager = $this->createMock(\Drupal\Core\Entity\EntityTypeManagerInterface::class);
        $attribute_manager = $this->createMock(\Drupal\sogan_commerce_product\Service\AttributeManager::class);
        $stock_service_manager = $this->createMock(\Drupal\commerce_stock\StockServiceManager::class);
        $remote_media_manager = $this->createMock(\Drupal\sogan_commerce_product\Service\RemoteMediaManager::class);
        return new ProductBuilder($entity_type_manager, $attribute_manager, $stock_service_manager, $remote_media_manager);
    }

    public function testComputeSingleNode()
    {
        $pb = $this->createProductBuilderWithMocks();

        $term = new class {
            private $fields;
            public function __construct()
            {
                $this->fields = [
                    'field_city_code' => new class {
                        public $value = '34';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_district_code' => new class {
                        public $value = 'K';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_supplier_code' => new class {
                        public $value = 'A';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                ];
            }
            public function hasField($name)
            {
                return isset($this->fields[$name]);
            }
            public function get($name)
            {
                return $this->fields[$name];
            }
        };

        $node = new \stdClass();
        $node->field_supplier = (object) ['entity' => $term];

        $body = $pb->computeLogisticsCodeBodyFromSupplierNodes([$node]);
        $this->assertEquals('34KA', $body);
    }

    public function testComputeMultipleDifferentCities()
    {
        $pb = $this->createProductBuilderWithMocks();

        $term1 = new class {
            private $fields;
            public function __construct()
            {
                $this->fields = [
                    'field_city_code' => new class {
                        public $value = '34';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_district_code' => new class {
                        public $value = 'K';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_supplier_code' => new class {
                        public $value = 'A';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                ];
            }
            public function hasField($name)
            {
                return isset($this->fields[$name]);
            }
            public function get($name)
            {
                return $this->fields[$name];
            }
        };

        $term2 = new class {
            private $fields;
            public function __construct()
            {
                $this->fields = [
                    'field_city_code' => new class {
                        public $value = '35';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_district_code' => new class {
                        public $value = 'L';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_supplier_code' => new class {
                        public $value = 'B';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                ];
            }
            public function hasField($name)
            {
                return isset($this->fields[$name]);
            }
            public function get($name)
            {
                return $this->fields[$name];
            }
        };

        $node1 = new \stdClass();
        $node1->field_supplier = (object) ['entity' => $term1];
        $node2 = new \stdClass();
        $node2->field_supplier = (object) ['entity' => $term2];

        $body = $pb->computeLogisticsCodeBodyFromSupplierNodes([$node1, $node2]);
        $this->assertEquals('99XX', substr($body, 0, 4)); // city/district/supplier parts are in first 4 char? actually 4 chars in our format
        // More explicitly, city part should be '99'
        $this->assertEquals('99', substr($body, 0, 2));
        $this->assertEquals('X', substr($body, 2, 1));
        $this->assertEquals('X', substr($body, 3, 1));
    }

    public function testSetVariationLogisticsCodeWithTermWrapper()
    {
        $pb = $this->createProductBuilderWithMocks();

        $term = new class {
            private $fields;
            public function __construct()
            {
                $this->fields = [
                    'field_city_code' => new class {
                        public $value = '34';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_district_code' => new class {
                        public $value = 'K';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                    'field_supplier_code' => new class {
                        public $value = 'A';
                        public function isEmpty()
                        {
                            return FALSE;
                        }
                    },
                ];
            }
            public function hasField($name)
            {
                return isset($this->fields[$name]);
            }
            public function get($name)
            {
                return $this->fields[$name];
            }
        };

        $variation = new class {
            private $field_logistics_code;
            private $id = 12;
            public function hasField($name)
            {
                return $name === 'field_logistics_code';
            }
            public function set($name, $value)
            {
                $this->{$name} = $value;
            }
            public function id()
            {
                return $this->id;
            }
            public function save()
            { /* no-op */
            }
            public function get($name)
            {
                return $this->{$name} ?? NULL;
            }
        };

        $pb->setVariationLogisticsCode($variation, $term);
        $code = $variation->get('field_logistics_code');
        $this->assertStringStartsWith('34K', $code);
        $this->assertStringEndsWith(str_pad(12, 6, '0', STR_PAD_LEFT), $code);
    }
}
