<?php

namespace Drupal\Tests\sogan_commerce_product\Kernel\Action;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\commerce_price\Price;
use Drupal\commerce_product\Entity\Product;

/**
 * Tests the CreateCommerceProduct action.
 *
 * @group sogan_commerce_product
 */
class CreateCommerceProductTest extends KernelTestBase
{
    use \Drupal\commerce_store\StoreCreationTrait;

    /**
     * Modules to enable.
     *
     * @var array
     */
    protected static $modules = [
        'system',
        'user',
        'node',
        'field',
        'text',
        'filter',
        'path',
        'path_alias',
        'entity',
        'token',
        'file',
        'image',
        'media',
        'media_library',
        'views',
        'entity_test',
        'commerce',
        'commerce_price',
        'commerce_store',
        'commerce_order',
        'commerce_product',
        'taxonomy',
        'options',
        'address',
        'datetime',
        'inline_entity_form',
        'commerce_stock',
        'commerce_stock_local',
        'commerce_stock_field',
        'address_tr',
        'entity_reference_revisions',
        'state_machine',
        'profile',
        'sogan_commerce_product',
    ];

    /**
     * Disable strict config schema checking.
     *
     * @var bool
     */
    protected $strictConfigSchema = FALSE;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->installEntitySchema('user');
        $this->installEntitySchema('file');
        $this->installEntitySchema('node');
        $this->installEntitySchema('path_alias');
        $this->installEntitySchema('taxonomy_term');
        $this->installEntitySchema('commerce_product');
        $this->installEntitySchema('commerce_product_variation');
        $this->installEntitySchema('commerce_store');
        $this->installEntitySchema('commerce_currency');
        $this->installEntitySchema('commerce_order');
        $this->installEntitySchema('commerce_order_item');
        $this->installEntitySchema('commerce_stock_location');
        $this->installEntitySchema('commerce_stock_location_type');

        // Install required config
        // Install base config first
        $this->installConfig(['system', 'file', 'image', 'commerce_product', 'commerce_stock', 'commerce_store']);

        // Import Currencies
        $currency_importer = $this->container->get('commerce_price.currency_importer');
        $currency_importer->import('TRY');
        $currency_importer->import('USD');

        // Create Default Store
        $this->createStore('Default Store', 'admin@example.com', 'default', TRUE, 'TR', 'TRY');

        // Create the 'suppliers' vocabulary
        if (!Vocabulary::load('suppliers')) {
            Vocabulary::create([
                'vid' => 'suppliers',
                'name' => 'Suppliers',
            ])->save();
        }

        // Create the 'brands' vocabulary
        if (!Vocabulary::load('brands')) {
            Vocabulary::create([
                'vid' => 'brands',
                'name' => 'Brands',
            ])->save();
        }

        // Install sogan_commerce_product config AFTER vocabularies exist
        $this->installConfig(['sogan_commerce_product']);

        // Configure commerce_stock service manager
        $this->config('commerce_stock.service_manager')
            ->set('default_service_id', 'local_stock')
            ->save();
        $this->installConfig(['commerce_stock_local']);
        $this->installConfig(['commerce_stock_field']);

        // Create 'supplier_product' content type
        if (!\Drupal::entityTypeManager()->getStorage('node_type')->load('supplier_product')) {
            \Drupal::entityTypeManager()->getStorage('node_type')->create([
                'type' => 'supplier_product',
                'name' => 'Supplier Product',
            ])->save();
        }

        // Create fields on 'supplier_product'
        $this->createField('node', 'supplier_product', 'field_list_price', 'commerce_price', 'commerce_price_default');
        $this->createField('node', 'supplier_product', 'field_cost_price', 'commerce_price', 'commerce_price_default');
        $this->createField('node', 'supplier_product', 'field_suggested_price', 'commerce_price', 'commerce_price_default');
        $this->createField('node', 'supplier_product', 'field_stock', 'integer', 'number');
        $this->createField('node', 'supplier_product', 'field_vat_rate', 'float', 'number');

        $this->createField('node', 'supplier_product', 'field_supplier_sku', 'string', 'string_textfield');
        $this->createField('node', 'supplier_product', 'field_global_unique_id', 'string', 'string_textfield');
        $this->createField('node', 'supplier_product', 'field_supplier_product_descripti', 'text_long', 'text_textarea');

        // field_supplier entity reference
        $this->createField('node', 'supplier_product', 'field_supplier', 'entity_reference', 'options_select', ['target_type' => 'taxonomy_term']);

        // Create field_source_supplier_products on commerce_product_variation
        $this->createField('commerce_product_variation', 'default', 'field_source_supplier_products', 'entity_reference', 'options_select', ['target_type' => 'node']);
    }

    /**
     * Helper to create a field.
     */
    protected function createField($entity_type, $bundle, $field_name, $field_type, $widget_type, $storage_settings = [])
    {
        if (!\Drupal\field\Entity\FieldStorageConfig::loadByName($entity_type, $field_name)) {
            \Drupal\field\Entity\FieldStorageConfig::create([
                'field_name' => $field_name,
                'entity_type' => $entity_type,
                'type' => $field_type,
                'settings' => $storage_settings,
            ])->save();
        }

        if (!\Drupal\field\Entity\FieldConfig::loadByName($entity_type, $bundle, $field_name)) {
            \Drupal\field\Entity\FieldConfig::create([
                'field_name' => $field_name,
                'entity_type' => $entity_type,
                'bundle' => $bundle,
                'label' => $field_name,
            ])->save();
        }
    }

    /**
     * Test execution of the Create Commerce Product action.
     */
    public function testCreateProductAction()
    {
        // 1. Create a Supplier Term
        $supplier = Term::create([
            'vid' => 'suppliers',
            'name' => 'Test Supplier',
            'field_supplier_weight' => 0,
        ]);
        $supplier->save();

        // 2. Create a Supplier Product Node
        $node = Node::create([
            'type' => 'supplier_product',
            'title' => 'Test Supplier Product',
            'field_supplier_product_descripti' => [['value' => 'Test Description', 'format' => 'plain_text']],
            'field_supplier' => $supplier->id(),
            'field_supplier_sku' => 'TEST-SKU-001',
            'field_list_price' => ['number' => '120.00', 'currency_code' => 'TRY'], // List price
            'field_suggested_price' => ['number' => '100.00', 'currency_code' => 'TRY'], // Selling price
            'field_global_unique_id' => 'TEST-GUID-001',
        ]);
        $node->save();

        // 3. Load the Action Plugin
        /** @var \Drupal\Core\Action\ActionManager $action_manager */
        $action_manager = $this->container->get('plugin.manager.action');

        /** @var \Drupal\sogan_commerce_product\Plugin\Action\CreateCommerceProduct $action */
        $action = $action_manager->createInstance('sogan_commerce_create_product_from_supplier_node');
        $this->assertNotNull($action, 'Action instance should be created.');

        // 4. Execute the action
        $action->execute($node);

        // 5. Assert: Check if a product was created
        $products = \Drupal::entityTypeManager()->getStorage('commerce_product')->loadMultiple();
        $this->assertNotEmpty($products, 'A commerce product should have been created.');

        $product = reset($products);
        $this->assertEquals('Test Supplier Product', $product->getTitle());

        // Check variation
        $variations = $product->getVariations();
        $this->assertNotEmpty($variations, 'Product should have variations.');
        $variation = reset($variations);
        $this->assertEquals(100.00, (float) $variation->getPrice()->getNumber());
    }
}
