<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\node\NodeInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for creating and managing product variations from supplier products.
 */
class VariationBuilder
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The attribute manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\AttributeManager
     */
    protected $attributeManager;

    /**
     * The attribute resolver.
     *
     * @var \Drupal\sogan_commerce_product\Service\AttributeResolver
     */
    protected $attributeResolver;

    /**
     * The price calculator.
     *
     * @var \Drupal\sogan_commerce_product\Service\PriceCalculator
     */
    protected $priceCalculator;

    /**
     * The tax rate resolver.
     *
     * @var \Drupal\sogan_commerce_product\Service\TaxRateResolver
     */
    protected $taxRateResolver;

    /**
     * The stock synchronizer.
     *
     * @var \Drupal\sogan_commerce_product\Service\StockSynchronizer
     */
    protected $stockSynchronizer;

    /**
     * The logistics code generator.
     *
     * @var \Drupal\sogan_commerce_product\Service\LogisticsCodeGenerator
     */
    protected $logisticsCodeGenerator;

    /**
     * The stock manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\StockManager
     */
    protected $stockManager;

    /**
     * The config.
     *
     * @var \Drupal\Core\Config\ImmutableConfig
     */
    protected $config;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * The transaction manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\TransactionManager
     */
    protected $transactionManager;

    /**
     * Constructs a VariationBuilder object.
     */
    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        AttributeManager $attribute_manager,
        AttributeResolver $attribute_resolver,
        PriceCalculator $price_calculator,
        TaxRateResolver $tax_rate_resolver,
        StockSynchronizer $stock_synchronizer,
        LogisticsCodeGenerator $logistics_code_generator,
        StockManager $stock_manager,
        ConfigFactoryInterface $config_factory,
        LoggerChannelFactoryInterface $logger_factory,
        TransactionManager $transaction_manager
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->attributeManager = $attribute_manager;
        $this->attributeResolver = $attribute_resolver;
        $this->priceCalculator = $price_calculator;
        $this->taxRateResolver = $tax_rate_resolver;
        $this->stockSynchronizer = $stock_synchronizer;
        $this->logisticsCodeGenerator = $logistics_code_generator;
        $this->stockManager = $stock_manager;
        $this->config = $config_factory->get('sogan_commerce_product.settings');
        $this->logger = $logger_factory->get('sogan_commerce_product');
        $this->transactionManager = $transaction_manager;
    }

    /**
     * Create a single variation from a supplier node.
     *
     * @param \Drupal\node\NodeInterface $supplier_node
     *   The supplier product node.
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product.
     * @param array $variation_attribute_keys
     *   Attribute keys to extract.
     *
     * @return \Drupal\commerce_product\Entity\ProductVariationInterface
     *   The created variation.
     */
    public function createVariation(NodeInterface $supplier_node, ProductInterface $product, array $variation_attribute_keys): ProductVariationInterface
    {
        return $this->transactionManager->executeInTransaction(function () use ($supplier_node, $product, $variation_attribute_keys) {
            // Determine SKU mode
            $mode = $this->config->get('visibility_mode') ?? 'readonly';
            $sku = NULL;
            if ($mode === 'do_nothing') {
                if ($supplier_node->hasField('field_supplier_sku') && !$supplier_node->get('field_supplier_sku')->isEmpty()) {
                    $sku = (string) $supplier_node->get('field_supplier_sku')->value;
                } elseif ($supplier_node->hasField('field_sku') && !$supplier_node->get('field_sku')->isEmpty()) {
                    $sku = (string) $supplier_node->get('field_sku')->value;
                }
            }

            // Gather price fields
            $price_value = $this->extractPriceField($supplier_node, 'field_suggested_price');
            $list_price_value = $this->extractPriceField($supplier_node, 'field_list_price');
            $cost_price_value = $this->extractPriceField($supplier_node, 'field_cost_price');

            $variation_data = [
                'type' => 'default',
                'title' => $supplier_node->getTitle(),
                'product_id' => $product->id(),
                'field_source_supplier_products' => [$supplier_node->id()],
            ];
            if (!empty($sku)) {
                $variation_data['sku'] = $sku;
            }
            if ($price_value) {
                $variation_data['price'] = $price_value;
            }
            if ($list_price_value) {
                $variation_data['list_price'] = $list_price_value;
            }
            if ($cost_price_value) {
                $variation_data['field_cost_price'] = $cost_price_value;
            }

            $variation = ProductVariation::create($variation_data);

            // Ensure local stock tracking
            if ($variation->hasField('commerce_stock_always_in_stock')) {
                $variation->set('commerce_stock_always_in_stock', 0);
            }

            // Set attributes
            $attributes_data = [];
            foreach ($variation_attribute_keys as $key) {
                $value = $this->attributeResolver->getAttributeValueFromNode($supplier_node, $key);
                if ($value) {
                    $attributes_data[] = ['name' => $key, 'value' => $value];
                }
            }
            $variation = $this->attributeManager->setVariationAttributes($variation, $attributes_data);

            // Set tax rate
            $tax_rate_id = $this->taxRateResolver->calculateTaxRateFromSuppliers([$supplier_node]);
            if ($tax_rate_id !== NULL && $variation->hasField('field_tax_rate')) {
                $variation->set('field_tax_rate', ['value' => $tax_rate_id]);
            }

            $variation->save();

            // Set back-reference on supplier node
            if ($supplier_node->hasField('field_associated_product_variati')) {
                $supplier_node->set('field_associated_product_variati', $variation->id());
                $supplier_node->save();
            }

            // Handle stock
            $location = $this->stockManager->getSupplierLocation($supplier_node);
            if ($location) {
                $new_quantity = $this->stockManager->getSupplierStock($supplier_node);
                if ($new_quantity > 0) {
                    $this->stockManager->setStockLevel(
                        $variation,
                        $location,
                        $new_quantity,
                        'Stock update from supplier node: ' . $supplier_node->id()
                    );
                }
            }

            return $variation;
        }, 'create_variation');
    }

    /**
     * Create a variation from a supplier group.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product.
     * @param array $group
     *   Group array with 'nodes', 'original_attrs', 'normalized_attrs'.
     *
     * @return \Drupal\commerce_product\Entity\ProductVariationInterface
     *   The created variation.
     */
    public function createVariationFromGroup(ProductInterface $product, array $group): ProductVariationInterface
    {
        return $this->transactionManager->executeInTransaction(function () use ($product, $group) {
            $supplier_nodes = $group['nodes'];

            $variation = ProductVariation::create([
                'type' => 'default',
                'sku' => 'SKU-' . uniqid(),
                'product_id' => $product->id(),
            ]);

            if ($variation->hasField('commerce_stock_always_in_stock')) {
                $variation->set('commerce_stock_always_in_stock', 0);
            }

            // Set supplier references
            $variation->set('field_source_supplier_products', array_map(function ($node) {
                return ['target_id' => $node->id()];
            }, $supplier_nodes));

            // Calculate and set prices
            $this->setPricesFromSuppliers($variation, $supplier_nodes);

            // Set tax rate
            $tax_rate_id = $this->taxRateResolver->calculateTaxRateFromSuppliers($supplier_nodes);
            if ($tax_rate_id !== NULL && $variation->hasField('field_tax_rate')) {
                $variation->set('field_tax_rate', ['value' => $tax_rate_id]);
            }

            // Create commerce attributes
            if (!empty($group['original_attrs'])) {
                $variation = $this->assignAttributesToVariation($variation, $group['original_attrs']);
            }

            // Set logistics code
            $this->logisticsCodeGenerator->setVariationLogisticsCodeFromSupplierNodes($variation, $supplier_nodes);

            $variation->save();

            // Add to product
            $product->addVariation($variation);
            $product->save();

            // Sync stock
            $this->stockSynchronizer->setVariationStocksGroupedBySupplier($variation, $supplier_nodes);

            return $variation;
        }, 'create_variation_from_group');
    }

    /**
     * Update an existing variation with new supplier sources.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation.
     * @param array $new_supplier_nodes
     *   New supplier nodes.
     */
    public function updateVariationWithSuppliers(ProductVariationInterface $variation, array $new_supplier_nodes): void
    {
        $this->transactionManager->executeInTransaction(function () use ($variation, $new_supplier_nodes) {
            // Get existing supplier references
            $existing_refs = [];
            if ($variation->hasField('field_source_supplier_products') && !$variation->get('field_source_supplier_products')->isEmpty()) {
                foreach ($variation->get('field_source_supplier_products') as $item) {
                    $existing_refs[$item->target_id] = $item->target_id;
                }
            }

            // Add new suppliers
            foreach ($new_supplier_nodes as $node) {
                $existing_refs[$node->id()] = $node->id();
            }

            // Load all supplier nodes
            $node_storage = $this->entityTypeManager->getStorage('node');
            $all_supplier_nodes = array_values($node_storage->loadMultiple($existing_refs));

            // Update field_source_supplier_products
            $variation->set('field_source_supplier_products', array_map(function ($node) {
                return ['target_id' => $node->id()];
            }, $all_supplier_nodes));

            // Recalculate prices
            $this->setPricesFromSuppliers($variation, $all_supplier_nodes);

            // Update tax rate
            $tax_rate_id = $this->taxRateResolver->calculateTaxRateFromSuppliers($all_supplier_nodes);
            if ($tax_rate_id !== NULL && $variation->hasField('field_tax_rate')) {
                $variation->set('field_tax_rate', ['value' => $tax_rate_id]);
            }

            // Update logistics code
            $this->logisticsCodeGenerator->setVariationLogisticsCodeFromSupplierNodes($variation, $all_supplier_nodes);

            // Ensure tracked stock mode
            if ($variation->hasField('commerce_stock_always_in_stock')) {
                $variation->set('commerce_stock_always_in_stock', 0);
                $variation->save();
            }

            // Update stock
            $this->stockSynchronizer->setVariationStocksGroupedBySupplier($variation, $all_supplier_nodes);

            $variation->save();
        }, 'update_variation_with_suppliers');
    }

    /**
     * Extract commerce attributes from a variation as normalized map.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation.
     *
     * @return array
     *   Normalized attribute map ['color' => 'red', 'size' => 'l'].
     */
    public function extractVariationAttributes(ProductVariationInterface $variation): array
    {
        $normalized = [];
        $field_definitions = $variation->getFieldDefinitions();

        foreach ($field_definitions as $field_name => $field_definition) {
            if (strpos($field_name, 'attribute_') !== 0) {
                continue;
            }

            if ($variation->hasField($field_name) && !$variation->get($field_name)->isEmpty()) {
                $attribute_value = $variation->get($field_name)->entity;
                if ($attribute_value) {
                    $attribute_label = substr($field_name, strlen('attribute_'));
                    $value_label = $attribute_value->label();
                    $norm_label = $this->attributeResolver->normalizeAttributeLabel($attribute_label);
                    $norm_value = $this->attributeResolver->normalizeAttributeValue($value_label);
                    $normalized[$norm_label] = $norm_value;
                }
            }
        }

        return $normalized;
    }

    /**
     * Check if a variation matches the given normalized attributes.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation.
     * @param array $normalized_attrs
     *   Normalized attribute map.
     *
     * @return bool
     *   TRUE if attributes match.
     */
    public function matchVariationByAttributes(ProductVariationInterface $variation, array $normalized_attrs): bool
    {
        $variation_attrs = $this->extractVariationAttributes($variation);
        $var_signature = $this->attributeResolver->createAttributeSignature($variation_attrs);
        $supplier_signature = $this->attributeResolver->createAttributeSignature($normalized_attrs);

        return $var_signature === $supplier_signature;
    }

    /**
     * Assign commerce attributes to a variation.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation.
     * @param array $attributes
     *   Attribute map ['Color' => 'Red', 'Size' => 'Large'].
     *
     * @return \Drupal\commerce_product\Entity\ProductVariationInterface
     *   Updated variation.
     */
    public function assignAttributesToVariation(ProductVariationInterface $variation, array $attributes): ProductVariationInterface
    {
        $items = [];
        foreach ($attributes as $label => $value) {
            $items[] = ['name' => $label, 'value' => $value];
        }
        if (!empty($items)) {
            $variation = $this->attributeManager->setVariationAttributes($variation, $items);
        }
        return $variation;
    }

    /**
     * Extract price field from supplier node.
     */
    protected function extractPriceField(NodeInterface $node, string $field_name): ?array
    {
        if ($node->hasField($field_name) && !$node->get($field_name)->isEmpty()) {
            $price = $node->get($field_name)->first()->getValue();
            return [
                'number' => $price['number'] ?? '0.00',
                'currency_code' => $price['currency_code'] ?? 'TRY',
            ];
        }
        return NULL;
    }

    /**
     * Set prices on variation from supplier nodes.
     */
    protected function setPricesFromSuppliers(ProductVariationInterface $variation, array $supplier_nodes): void
    {
        $cost_price = $this->priceCalculator->calculateCostPrice($supplier_nodes);
        $list_price = $this->priceCalculator->calculateListPrice($supplier_nodes);
        $selling_price = $this->priceCalculator->calculateSellingPrice($supplier_nodes);

        if ($cost_price !== NULL && $variation->hasField('field_cost_price')) {
            $variation->set('field_cost_price', [
                'number' => $cost_price,
                'currency_code' => 'TRY',
            ]);
        }

        if ($list_price !== NULL && $variation->hasField('list_price')) {
            $variation->set('list_price', [
                'number' => $list_price,
                'currency_code' => 'TRY',
            ]);
        }

        if ($selling_price !== NULL && $variation->hasField('price')) {
            $variation->set('price', [
                'number' => $selling_price,
                'currency_code' => 'TRY',
            ]);
        }
    }
}
