<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_stock\StockLocationInterface;

/**
 * Service to manage database transactions and locking for critical operations.
 */
class TransactionManager
{
  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new TransactionManager object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(Connection $database, LoggerChannelFactoryInterface $logger_factory)
  {
    $this->database = $database;
    $this->logger = $logger_factory->get('sogan_commerce_product');
  }

  /**
   * Execute a callback within a database transaction.
   *
   * @param callable $callback
   *   The callback to execute. Should return the operation result.
   * @param string $name
   *   Optional transaction name for debugging.
   *
   * @return mixed
   *   The return value of the callback.
   *
   * @throws \Throwable
   *   Any exception from the callback will cause automatic rollback.
   */
  public function executeInTransaction(callable $callback, string $name = 'sogan_commerce')
  {
    $transaction = $this->database->startTransaction($name);

    try {
      $result = $callback();
      // Transaction commits automatically when $transaction goes out of scope
      unset($transaction);
      return $result;
    } catch (\Throwable $e) {
      // Transaction rolls back automatically due to exception
      if (isset($transaction)) {
        unset($transaction);
      }
      // Re-throw to allow caller to handle
      throw $e;
    }
  }

  /**
   * Acquire a row-level lock for stock updates on a variation/location pair.
   *
   * This uses SELECT ... FOR UPDATE to ensure exclusive access during
   * stock level calculations to prevent race conditions.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param \Drupal\commerce_stock\StockLocationInterface $location
   *   The stock location.
   *
   * @return bool
   *   TRUE if lock acquired, FALSE otherwise.
   */
  public function acquireStockLock(ProductVariationInterface $variation, StockLocationInterface $location)
  {
    // Lock the transaction rows for this variation/location combination
    // This prevents concurrent modifications to the same stock level
    try {
      $query = $this->database->select('commerce_stock_transaction', 't');
      $query->fields('t', ['id']);
      $query->condition('entity_id', $variation->id());
      $query->condition('entity_type', $variation->getEntityTypeId());
      $query->condition('location_id', $location->id());
      $query->orderBy('id', 'DESC');
      $query->range(0, 1);
      // Add FOR UPDATE to lock the rows
      $query->forUpdate();

      // Execute the query to acquire the lock
      $query->execute();

      return TRUE;
    } catch (\Exception $e) {
      $this->logger->error('Failed to acquire stock lock: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Acquire a lock on the SKU sequence for a given prefix.
   *
   * @param string $prefix
   *   The SKU prefix to lock.
   *
   * @return bool
   *   TRUE if lock acquired successfully.
   */
  public function acquireSkuSequenceLock(string $prefix)
  {
    try {
      // Lock the sequence row for this prefix
      $query = $this->database->select('sogan_commerce_sku_sequence', 's');
      $query->fields('s', ['prefix', 'sequence']);
      $query->condition('prefix', $prefix);
      $query->forUpdate();

      $query->execute();
      return TRUE;
    } catch (\Exception $e) {
      $this->logger->error('Failed to acquire SKU sequence lock: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Check if a supplier node is already associated with a variation (with lock).
   *
   * This acquires a lock on the supplier node reference to prevent duplicate
   * variation creation during concurrent operations.
   *
   * @param int $supplier_node_id
   *   The supplier product node ID.
   *
   * @return int|null
   *   The variation ID if already associated, NULL otherwise.
   */
  public function getSupplierVariationWithLock(int $supplier_node_id)
  {
    try {
      // Lock the supplier node field row to prevent concurrent checks
      $query = $this->database->select('node__field_associated_product_variati', 'fa');
      $query->fields('fa', ['field_associated_product_variati_target_id']);
      $query->condition('entity_id', $supplier_node_id);
      $query->forUpdate();

      $result = $query->execute()->fetchField();

      return $result ? (int) $result : NULL;
    } catch (\Exception $e) {
      $this->logger->warning('Failed to check supplier variation with lock: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }
}
