<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\taxonomy\Entity\Term;

class TaxonomyManager
{
    protected $entityTypeManager;
    protected $logger;

    public function __construct(EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory)
    {
        $this->entityTypeManager = $entity_type_manager;
        $this->logger = $logger_factory->get('sogan_commerce_product');
    }

    /**
     * Resolve suggested values into term reference arrays (['target_id' => TID]).
     * Accepts arrays returned by field->getValue() (which may contain
     * 'target_id' or 'value') or plain strings. For hierarchical strings
     * using '>' separator (e.g. "Parent > Child"), create or resolve the
     * parent/child chain and return the child term.
     *
     * @param string $vocab
     *   Vocabulary machine name.
     * @param array|string $values
     *   Values to resolve. May be a single string or an array of mixed items.
     *
     * @return array
     *   Array of term reference arrays suitable for entity reference fields.
     */
    public function resolveSuggestedValues(string $vocab, $values): array
    {
        if (is_string($values)) {
            $values = [$values];
        }
        if (!is_array($values)) {
            return [];
        }

        $storage = $this->entityTypeManager->getStorage('taxonomy_term');
        $results = [];

        foreach ($values as $v) {
            // Normalize field->getValue() arrays
            if (is_array($v)) {
                if (isset($v['target_id'])) {
                    $tid = (int) $v['target_id'];
                    if ($tid > 0) {
                        $results[] = ['target_id' => $tid];
                        continue;
                    }
                }
                if (isset($v['tid'])) {
                    $tid = (int) $v['tid'];
                    if ($tid > 0) {
                        $results[] = ['target_id' => $tid];
                        continue;
                    }
                }
                if (isset($v['value'])) {
                    $v = $v['value'];
                } else {
                    // Fallback: try first scalar value in array
                    foreach ($v as $maybe) {
                        if (is_scalar($maybe)) {
                            $v = (string) $maybe;
                            break;
                        }
                    }
                }
            }

            if (!is_string($v)) {
                continue;
            }

            $str = trim($v);
            if ($str === '') continue;

            // Hierarchical categories: split by '>' and create chain.
            if (strpos($str, '>') !== FALSE) {
                $parts = array_map('trim', explode('>', $str));
                $parent_tid = 0;
                foreach ($parts as $part) {
                    $parent_tid = $this->getOrCreateTermByName($vocab, $part, $parent_tid);
                }
                if ($parent_tid) {
                    $results[] = ['target_id' => $parent_tid];
                }
                continue;
            }

            // Non-hierarchical: get or create term at top level.
            $tid = $this->getOrCreateTermByName($vocab, $str, 0);
            if ($tid) {
                $results[] = ['target_id' => $tid];
            }
        }

        // Deduplicate by target_id
        $seen = [];
        $out = [];
        foreach ($results as $r) {
            $tid = (int) ($r['target_id'] ?? 0);
            if ($tid && !isset($seen[$tid])) {
                $seen[$tid] = TRUE;
                $out[] = ['target_id' => $tid];
            }
        }
        return $out;
    }

    /**
     * Get an existing term by name and parent or create it.
     * Returns the term id.
     */
    protected function getOrCreateTermByName(string $vocab, string $name, int $parent_tid = 0): int
    {
        $storage = $this->entityTypeManager->getStorage('taxonomy_term');

        // Query for exact match with given parent.
        $query = $storage->getQuery();
        $query->condition('vid', $vocab);
        $query->condition('name', $name);
        // Parent field stores 0 for top-level items.
        $query->condition('parent', $parent_tid);
        // Access bypass intentional: Backend taxonomy lookup without user context.
        $query->accessCheck(FALSE);
        $tids = $query->execute();
        if (!empty($tids)) {
            $tid = reset($tids);
            return (int) $tid;
        }

        // Create term
        try {
            $values = ['vid' => $vocab, 'name' => $name];
            if ($parent_tid) {
                $values['parent'] = [$parent_tid];
            }
            $term = Term::create($values);
            $term->save();
            return (int) $term->id();
        } catch (\Throwable $e) {
            $this->logger->error('Failed to create term @name in vocab @v: @msg', ['@name' => $name, '@v' => $vocab, '@msg' => $e->getMessage()]);
            return 0;
        }
    }
}
