<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\node\NodeInterface;

/**
 * Service for resolving tax rates from supplier products.
 */
class TaxRateResolver
{

    /**
     * Calculate tax rate ID from multiple suppliers.
     *
     * Uses the highest VAT rate among all supplier products to determine
     * the appropriate tax rate ID for the variation.
     *
     * @param array $supplier_nodes
     *   Array of supplier product nodes.
     *
     * @return string|null
     *   Tax rate ID ('standard', 'reduced_high', 'reduced_low') or NULL.
     */
    public function calculateTaxRateFromSuppliers(array $supplier_nodes): ?string
    {
        if (empty($supplier_nodes)) {
            return NULL;
        }

        $max_vat_rate = NULL;

        foreach ($supplier_nodes as $node) {
            if (!$node instanceof NodeInterface) {
                continue;
            }
            if (!$node->hasField('field_vat_rate') || $node->get('field_vat_rate')->isEmpty()) {
                continue;
            }

            $vat_rate = (float) $node->get('field_vat_rate')->value;

            if ($max_vat_rate === NULL || $vat_rate > $max_vat_rate) {
                $max_vat_rate = $vat_rate;
            }
        }

        if ($max_vat_rate === NULL) {
            return NULL;
        }

        // Map VAT rate percentages to tax rate IDs based on Turkey VAT rates.
        // Standard: 20%, Reduced High: 10%, Reduced Low: 1%
        if ($max_vat_rate >= 15) {
            return 'standard';
        } elseif ($max_vat_rate >= 5) {
            return 'reduced_high';
        } else {
            return 'reduced_low';
        }
    }
}
