<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;

/**
 * Service to resolve supplier product fields for products and variations.
 */
class SupplierProductFieldResolver {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new SupplierProductFieldResolver.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Get supplier product field values for a variation.
   *
   * @param int $variation_id
   *   The variation ID.
   * @param string $field_name
   *   The field name to retrieve from supplier products.
   *
   * @return array
   *   Array of field values from all supplier products linked to this variation.
   *   Each element contains:
   *   - 'supplier_product_id': The supplier product node ID
   *   - 'supplier_product_title': The supplier product node title
   *   - 'field_value': The field value (could be scalar, array, or NULL)
   */
  public function getVariationSupplierField($variation_id, $field_name) {
    $results = [];

    // Load the variation
    $variation = $this->entityTypeManager
      ->getStorage('commerce_product_variation')
      ->load($variation_id);

    if (!$variation instanceof ProductVariationInterface) {
      return $results;
    }

    // Check if variation has the source supplier products field
    if (!$variation->hasField('field_source_supplier_products') || 
        $variation->get('field_source_supplier_products')->isEmpty()) {
      return $results;
    }

    // Get all supplier product nodes referenced by this variation
    $supplier_nodes = $variation->get('field_source_supplier_products')->referencedEntities();

    foreach ($supplier_nodes as $node) {
      $result = [
        'supplier_product_id' => $node->id(),
        'supplier_product_title' => $node->label(),
        'field_value' => NULL,
      ];

      // Check if the supplier product has the requested field
      if ($node->hasField($field_name) && !$node->get($field_name)->isEmpty()) {
        $field_value = $node->get($field_name)->getValue();
        
        // If it's a single-value field, extract the value
        if (count($field_value) === 1 && isset($field_value[0]['value'])) {
          $result['field_value'] = $field_value[0]['value'];
        } else {
          // Return the full field value array for multi-value or complex fields
          $result['field_value'] = $field_value;
        }
      }

      $results[] = $result;
    }

    return $results;
  }

  /**
   * Get supplier product field values for all variations of a product.
   *
   * @param int $product_id
   *   The product ID.
   * @param string $field_name
   *   The field name to retrieve from supplier products.
   *
   * @return array
   *   Array of results grouped by variation ID.
   *   Format: [
   *     'variation_123' => [
   *       'variation_id' => 123,
   *       'variation_title' => 'Product - Red',
   *       'supplier_fields' => [
   *         [
   *           'supplier_product_id' => 456,
   *           'supplier_product_title' => 'Supplier Product 1',
   *           'field_value' => 'value',
   *         ],
   *       ],
   *     ],
   *   ]
   */
  public function getProductSupplierField($product_id, $field_name) {
    $results = [];

    // Load the product
    $product = $this->entityTypeManager
      ->getStorage('commerce_product')
      ->load($product_id);

    if (!$product instanceof ProductInterface) {
      return $results;
    }

    // Get all variations of this product
    $variations = $product->getVariations();

    foreach ($variations as $variation) {
      $variation_key = 'variation_' . $variation->id();
      
      $results[$variation_key] = [
        'variation_id' => $variation->id(),
        'variation_title' => $variation->label(),
        'supplier_fields' => $this->getVariationSupplierField($variation->id(), $field_name),
      ];
    }

    return $results;
  }

  /**
   * Get all unique values for a field across all supplier products of a variation.
   *
   * @param int $variation_id
   *   The variation ID.
   * @param string $field_name
   *   The field name to retrieve.
   *
   * @return array
   *   Array of unique field values.
   */
  public function getVariationSupplierFieldUniqueValues($variation_id, $field_name) {
    $field_data = $this->getVariationSupplierField($variation_id, $field_name);
    $unique_values = [];

    foreach ($field_data as $item) {
      if ($item['field_value'] !== NULL) {
        // Handle both scalar and array values
        if (is_array($item['field_value'])) {
          foreach ($item['field_value'] as $value) {
            if (is_array($value)) {
              // For complex field values, serialize for comparison
              $serialized = serialize($value);
              if (!in_array($serialized, $unique_values)) {
                $unique_values[] = $serialized;
              }
            } else {
              if (!in_array($value, $unique_values)) {
                $unique_values[] = $value;
              }
            }
          }
        } else {
          if (!in_array($item['field_value'], $unique_values)) {
            $unique_values[] = $item['field_value'];
          }
        }
      }
    }

    return $unique_values;
  }

  /**
   * Get the first non-empty field value from supplier products of a variation.
   *
   * @param int $variation_id
   *   The variation ID.
   * @param string $field_name
   *   The field name to retrieve.
   *
   * @return mixed|null
   *   The first non-empty field value, or NULL if none found.
   */
  public function getVariationSupplierFieldFirst($variation_id, $field_name) {
    $field_data = $this->getVariationSupplierField($variation_id, $field_name);

    foreach ($field_data as $item) {
      if ($item['field_value'] !== NULL) {
        return $item['field_value'];
      }
    }

    return NULL;
  }

  /**
   * Get supplier product nodes for a variation.
   *
   * @param int $variation_id
   *   The variation ID.
   *
   * @return \Drupal\node\NodeInterface[]
   *   Array of supplier product nodes.
   */
  public function getVariationSupplierProducts($variation_id) {
    $variation = $this->entityTypeManager
      ->getStorage('commerce_product_variation')
      ->load($variation_id);

    if (!$variation instanceof ProductVariationInterface) {
      return [];
    }

    if (!$variation->hasField('field_source_supplier_products') || 
        $variation->get('field_source_supplier_products')->isEmpty()) {
      return [];
    }

    return $variation->get('field_source_supplier_products')->referencedEntities();
  }

  /**
   * Get all supplier product nodes for all variations of a product.
   *
   * @param int $product_id
   *   The product ID.
   *
   * @return array
   *   Array of supplier product nodes grouped by variation.
   *   Format: [
   *     'variation_123' => [node1, node2],
   *     'variation_124' => [node3],
   *   ]
   */
  public function getProductSupplierProducts($product_id) {
    $results = [];

    $product = $this->entityTypeManager
      ->getStorage('commerce_product')
      ->load($product_id);

    if (!$product instanceof ProductInterface) {
      return $results;
    }

    $variations = $product->getVariations();

    foreach ($variations as $variation) {
      $variation_key = 'variation_' . $variation->id();
      $results[$variation_key] = $this->getVariationSupplierProducts($variation->id());
    }

    return $results;
  }

}
