<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_stock\StockServiceManager;
use Drupal\commerce_stock\StockTransactionsInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\node\NodeInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for synchronizing stock between supplier products and variations.
 */
class StockSynchronizer
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The stock manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\StockManager
     */
    protected $stockManager;

    /**
     * The commerce stock service manager.
     *
     * @var \Drupal\commerce_stock\StockServiceManager
     */
    protected $stockServiceManager;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a StockSynchronizer object.
     */
    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        StockManager $stock_manager,
        StockServiceManager $stock_service_manager,
        LoggerChannelFactoryInterface $logger_factory
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->stockManager = $stock_manager;
        $this->stockServiceManager = $stock_service_manager;
        $this->logger = $logger_factory->get('sogan_commerce_product');
    }

    /**
     * Sync stock for a specific supplier to its location.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation.
     * @param \Drupal\node\NodeInterface $supplier_node
     *   The supplier product node.
     */
    public function syncStockForSupplier(ProductVariationInterface $variation, NodeInterface $supplier_node): void
    {
        // Get stock quantity from supplier node
        $stock_qty = (int) $this->stockManager->getSupplierStock($supplier_node);

        // Get supplier term to find stock location
        $supplier_term = NULL;
        if ($supplier_node->hasField('field_supplier') && !$supplier_node->get('field_supplier')->isEmpty()) {
            $supplier_term = $supplier_node->get('field_supplier')->entity;
        }

        if (!$supplier_term) {
            return; // Can't sync without supplier term
        }

        // Get stock location for this supplier
        $stock_location = NULL;
        if ($supplier_term->hasField('field_stock_location') && !$supplier_term->get('field_stock_location')->isEmpty()) {
            $stock_location = $supplier_term->get('field_stock_location')->entity;
        }

        if (!$stock_location) {
            $this->logger->warning(
                'No stock location found for supplier @supplier, skipping stock sync',
                ['@supplier' => $supplier_term->label()]
            );
            return;
        }

        // Use stock service to update stock level
        try {
            $stock_updater = $this->stockServiceManager->getService($variation)->getStockUpdater();
            $stock_updater->createTransaction(
                $variation,
                $stock_location->id(),
                '',
                $stock_qty,
                NULL,
                'TRY',
                StockTransactionsInterface::STOCK_IN,
                []
            );
        } catch (\Exception $e) {
            $this->logger->error(
                'Failed to sync stock for variation @var, supplier @supplier: @msg',
                [
                    '@var' => $variation->id(),
                    '@supplier' => $supplier_term->label(),
                    '@msg' => $e->getMessage(),
                ]
            );
        }
    }

    /**
     * Set stock levels for a variation by grouping supplier nodes per supplier.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation to update.
     * @param array $supplier_nodes
     *   Array of supplier product node entities.
     */
    public function setVariationStocksGroupedBySupplier(ProductVariationInterface $variation, array $supplier_nodes): void
    {
        if (empty($supplier_nodes)) {
            return;
        }

        // Group nodes by supplier term id.
        $groups = [];
        foreach ($supplier_nodes as $node) {
            if (!$node || !$node->hasField('field_supplier') || $node->get('field_supplier')->isEmpty()) {
                continue;
            }
            $term = $node->get('field_supplier')->entity;
            if (!$term) {
                continue;
            }
            $tid = $term->id();
            $groups[$tid]['nodes'][] = $node;
            $groups[$tid]['term'] = $term;
        }

        // For each supplier group, compute total stock and set location level.
        foreach ($groups as $tid => $group) {
            $nodes = $group['nodes'];
            $term = $group['term'];
            $sample = reset($nodes);
            if (!$sample) {
                continue;
            }

            $location = $this->stockManager->getSupplierLocation($sample);
            if (!$location) {
                $this->logger->warning(
                    'No stock location found for supplier term @tid while syncing variation @vid',
                    ['@tid' => $tid, '@vid' => $variation->id()]
                );
                continue;
            }

            $total = 0.0;
            foreach ($nodes as $n) {
                // If a supplier reports negative stock, treat it as 0.
                $qty = (float) $this->stockManager->getSupplierStock($n);
                if ($qty < 0.0) {
                    $qty = 0.0;
                }
                $total += $qty;
            }

            $message = sprintf(
                'Aggregated stock sync from supplier %s for variation %d',
                $term->label() ?? $tid,
                $variation->id()
            );
            $this->stockManager->setStockLevel($variation, $location, (float) $total, $message);
        }
    }
}
