<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_stock\StockServiceManager;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_stock\StockLocationInterface;
use Drupal\commerce_stock\StockTransactionsInterface;
use Drupal\node\NodeInterface;
use Drupal\Core\Database\Connection;

/**
 * Service to manage stock operations.
 */
class StockManager
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The stock service manager.
   *
   * @var \Drupal\commerce_stock\StockServiceManager
   */
  protected $stockServiceManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The transaction manager.
   *
   * @var \Drupal\sogan_commerce_product\Service\TransactionManager
   */
  protected $transactionManager;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new StockManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\commerce_stock\StockServiceManager $stock_service_manager
   *   The stock service manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\sogan_commerce_product\Service\TransactionManager $transaction_manager
   *   The transaction manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, StockServiceManager $stock_service_manager, Connection $database, TransactionManager $transaction_manager, LoggerChannelFactoryInterface $logger_factory)
  {
    $this->entityTypeManager = $entity_type_manager;
    $this->stockServiceManager = $stock_service_manager;
    $this->database = $database;
    $this->transactionManager = $transaction_manager;
    $this->logger = $logger_factory->get('sogan_commerce_product');
  }

  /**
   * Get stock quantity from a supplier product node.
   *
   * @param \Drupal\node\NodeInterface $supplier_node
   *   The supplier product node.
   *
   * @return float
   *   The stock quantity.
   */
  public function getSupplierStock(NodeInterface $supplier_node)
  {
    if ($supplier_node->hasField('field_supplier_stock') && !$supplier_node->get('field_supplier_stock')->isEmpty()) {
      return (float) $supplier_node->get('field_supplier_stock')->value;
    }
    return 0.0;
  }

  /**
   * Get the stock location associated with a supplier product node.
   *
   * @param \Drupal\node\NodeInterface $supplier_node
   *   The supplier product node.
   *
   * @return \Drupal\commerce_stock\StockLocationInterface|null
   *   The stock location or NULL if not found.
   */
  public function getSupplierLocation(NodeInterface $supplier_node)
  {
    if (!$supplier_node->hasField('field_supplier') || $supplier_node->get('field_supplier')->isEmpty()) {
      return NULL;
    }

    $supplier_term = $supplier_node->get('field_supplier')->entity;
    if (!$supplier_term) {
      return NULL;
    }

    // 1. Try to get location from field reference on the taxonomy term
    if ($supplier_term->hasField('field_stock_location') && !$supplier_term->get('field_stock_location')->isEmpty()) {
      $location_id = $supplier_term->get('field_stock_location')->target_id;
      return $this->entityTypeManager->getStorage('commerce_stock_location')->load($location_id);
    }

    // 2. Fallback: Try to find location by name matching supplier name
    $supplier_name = $supplier_term->label();
    $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
    $locations = $location_storage->loadByProperties(['name' => $supplier_name]);

    if (!empty($locations)) {
      return reset($locations);
    }

    return NULL;
  }

  /**
   * Get current stock level for a variation at a specific location.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param \Drupal\commerce_stock\StockLocationInterface $location
   *   The stock location.
   *
   * @return float
   *   The current stock level.
   */
  public function getStockLevel(ProductVariationInterface $variation, StockLocationInterface $location)
  {
    $stock_service = $this->stockServiceManager->getService($variation);
    $stock_checker = $stock_service->getStockChecker();
    $location_id = $location->id();

    // Method 1: getLocationsStockLevels (Preferred)
    if (method_exists($stock_checker, 'getLocationsStockLevels')) {
      $levels = $stock_checker->getLocationsStockLevels($variation, [$location]);
      if (isset($levels[$location_id])) {
        return (float) $levels[$location_id]['qty'] + (float) $levels[$location_id]['transactions_qty'];
      }
    }

    // Method 2: getLocationStockLevel (Fallback)
    if (method_exists($stock_checker, 'getLocationStockLevel')) {
      $data = $stock_checker->getLocationStockLevel($location_id, $variation);

      $base_qty = isset($data['qty']) ? (float) $data['qty'] : 0.0;
      $transactions_qty = 0.0;

      // Calculate transactions since last snapshot
      if (
        method_exists($stock_checker, 'getLocationStockTransactionLatest') &&
        method_exists($stock_checker, 'getLocationStockTransactionSum')
      ) {
        $latest_txn = $stock_checker->getLocationStockTransactionLatest($location_id, $variation);
        $last_transaction = $data['last_transaction'] ?? 0;
        $transactions_qty = $stock_checker->getLocationStockTransactionSum($location_id, $variation, $last_transaction, $latest_txn);
      }

      return $base_qty + (float) $transactions_qty;
    }

    return 0.0;
  }

  /**
   * Create a stock transaction.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param \Drupal\commerce_stock\StockLocationInterface $location
   *   The stock location.
   * @param float $quantity
   *   The quantity to add/remove.
   * @param int $transaction_type
   *   The transaction type (STOCK_IN, STOCK_OUT, etc.).
   * @param string $message
   *   Optional message.
   *
   * @return int|false
   *   The transaction ID or FALSE on failure.
   */
  public function createTransaction(ProductVariationInterface $variation, StockLocationInterface $location, $quantity, $transaction_type, $message = '')
  {
    // Ensure we are using a service that supports transactions
    $stock_service = $this->stockServiceManager->getService($variation);
    $stock_updater = $stock_service->getStockUpdater();

    // Check if we can create transactions
    if (!$stock_updater) {
      $this->logger->error('Stock updater not available for variation @id', ['@id' => $variation->id()]);
      return FALSE;
    }

    try {
      // Call the underlying stock updater directly so we can capture and
      // return the created transaction id. The StockServiceManager wrapper
      // (from contrib/commerce_stock) does not currently return the id—this
      // caused our callers to receive NULL even when the transaction was
      // successfully created.
      $transaction_id = $stock_updater->createTransaction(
        $variation,
        $location->id(),
        '', // zone
        $quantity,
        0, // unit cost
        'TRY', // currency
        $transaction_type,
        ['message' => $message]
      );

      // Ensure we return whatever the updater returned (tid or FALSE).
      if (empty($transaction_id)) {
        // Log the updater class so we can debug why it did not create a
        // transaction (for example the AlwaysInStock updater returns NULL).

        $service_class = is_object($stock_updater) ? get_class($stock_updater) : 'none';

        // Log at debug level to help later diagnostics without spamming
        $this->logger->debug('Stock updater of class @class returned @value for variation @vid', [
          '@class' => $service_class,
          '@value' => var_export($transaction_id, TRUE),
          '@vid' => $variation->id(),
        ]);
      }

      return $transaction_id;
    } catch (\Exception $e) {
      $this->logger->error('Failed to create stock transaction: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

  /**
   * Set the stock level to a specific quantity.
   * Calculates the difference and creates a transaction.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param \Drupal\commerce_stock\StockLocationInterface $location
   *   The stock location.
   * @param float $target_quantity
   *   The desired stock quantity.
   * @param string $message
   *   Optional message.
   *
   * @return int|false
   *   The transaction ID or FALSE if no change needed or failure.
   */
  public function setStockLevel(ProductVariationInterface $variation, StockLocationInterface $location, $target_quantity, $message = '')
  {
    // Wrap in transaction with row-level locking to prevent race conditions
    return $this->transactionManager->executeInTransaction(function () use ($variation, $location, $target_quantity, $message) {
      // Acquire lock on stock transactions for this variation/location
      $this->transactionManager->acquireStockLock($variation, $location);

      // Now safely read current level (locked, no concurrent modifications possible)
      $current_level = $this->getStockLevel($variation, $location);
      $diff = $target_quantity - $current_level;

      if (abs($diff) < 0.001) {
        return FALSE; // No change needed
      }

      // Determine transaction type
      $transaction_type = ($diff > 0) ? StockTransactionsInterface::STOCK_IN : StockTransactionsInterface::STOCK_OUT;

      // Check if this is the first transaction (NEW_STOCK)
      $has_transactions = $this->hasTransactions($variation, $location);
      if (!$has_transactions && $diff > 0) {
        $transaction_type = StockTransactionsInterface::NEW_STOCK;
      }

      return $this->createTransaction($variation, $location, $diff, $transaction_type, $message);
    }, 'stock_set_level');
  }

  /**
   * Check if any transactions exist for the variation and location.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param \Drupal\commerce_stock\StockLocationInterface $location
   *   The stock location.
   *
   * @return bool
   *   TRUE if transactions exist.
   */
  public function hasTransactions(ProductVariationInterface $variation, StockLocationInterface $location)
  {
    $query = $this->database->select('commerce_stock_transaction', 't');
    $query->condition('entity_id', $variation->id());
    $query->condition('entity_type', $variation->getEntityTypeId());
    $query->condition('location_id', $location->id());
    $count = $query->countQuery()->execute()->fetchField();
    return $count > 0;
  }

  /**
   * Get recent transactions for a variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param \Drupal\commerce_stock\StockLocationInterface|null $location
   *   Optional stock location to filter by.
   * @param int $limit
   *   Number of transactions to return.
   *
   * @return array
   *   Array of transaction objects.
   */
  public function getTransactions(ProductVariationInterface $variation, ?StockLocationInterface $location = NULL, $limit = 10)
  {
    $query = $this->database->select('commerce_stock_transaction', 't');
    $query->fields('t');
    $query->condition('entity_id', $variation->id());
    $query->condition('entity_type', $variation->getEntityTypeId());

    if ($location) {
      $query->condition('location_id', $location->id());
    }

    $query->orderBy('id', 'DESC');
    $query->range(0, $limit);

    return $query->execute()->fetchAll();
  }
}
