<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;

/**
 * Provides incremental SKU sequence numbers per prefix.
 */
class SkuSequenceManager
{
    protected Connection $database;

    /**
     * The transaction manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\TransactionManager
     */
    protected $transactionManager;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    public function __construct(Connection $database, TransactionManager $transaction_manager, ConfigFactoryInterface $config_factory)
    {
        $this->database = $database;
        $this->transactionManager = $transaction_manager;
        $this->configFactory = $config_factory;
    }

    /**
     * Returns the next numeric sequence for the given prefix in a transaction.
     *
     * @param string $prefix
     *   The SKU prefix (e.g., 'P-').
     *
     * @return int
     *   The next numeric sequence for the prefix.
     */
    public function getNextSequence(string $prefix): int
    {
        $connection = $this->database;
        // Start transaction for safe increment.
        $transaction = $connection->startTransaction('sogan_commerce_product_sku_sequence');
        try {
            // Use SELECT FOR UPDATE to lock the row.
            $query = $connection->query("SELECT last FROM {sogan_commerce_sku_sequence} WHERE prefix = :prefix FOR UPDATE", [':prefix' => $prefix]);
            $row = $query->fetchAssoc();
            if ($row) {
                $last = (int) $row['last'];
                // Optionally consider existing SKUs to bring sequence forward
                // if manual SKUs were added after queries created the sequence.
                $config = $this->configFactory->get('sogan_commerce_product.sku_settings');
                $init_from_existing = (bool) ($config->get('sequence_init_from_existing') ?? FALSE);
                if ($init_from_existing) {
                    $prefix_length = strlen($prefix);
                    $like = $prefix . '%';
                    $regex = '^' . preg_quote($prefix, '/') . '[0-9]+$';
                    $sql = "SELECT MAX(CAST(SUBSTRING(sku, :pos) AS UNSIGNED)) AS maxnum FROM {commerce_product_variation_field_data} WHERE sku LIKE :like AND sku REGEXP :regex";
                    $maxnum = (int) $connection->query($sql, [':pos' => $prefix_length + 1, ':like' => $like, ':regex' => $regex])->fetchField();
                } else {
                    $maxnum = 0;
                }
                $candidate = max($last + 1, ($maxnum ? $maxnum + 1 : 0));
                $next = $candidate ?: ($last + 1);
                $connection->update('sogan_commerce_sku_sequence')
                    ->fields(['last' => $next])
                    ->condition('prefix', $prefix)
                    ->execute();
            } else {
                // If there is no row for this prefix, optionally initialize
                // the new sequence from the highest existing numeric SKU
                // matching the prefix, if the site configuration requests it.
                $config = $this->configFactory->get('sogan_commerce_product.sku_settings');
                $init_from_existing = (bool) ($config->get('sequence_init_from_existing') ?? FALSE);
                if ($init_from_existing) {
                    $prefix_length = strlen($prefix);
                    // Compute max numeric portion of SKU that matches the pattern.
                    $like = $prefix . '%';
                    // REGEXP: '^prefix[0-9]+$'
                    $regex = '^' . preg_quote($prefix, '/') . '[0-9]+$';
                    $sql = "SELECT MAX(CAST(SUBSTRING(sku, :pos) AS UNSIGNED)) AS maxnum FROM {commerce_product_variation_field_data} WHERE sku LIKE :like AND sku REGEXP :regex";
                    $maxnum = (int) $connection->query($sql, [':pos' => $prefix_length + 1, ':like' => $like, ':regex' => $regex])->fetchField();
                    $next = $maxnum ? ($maxnum + 1) : 1;
                } else {
                    $next = 1;
                }

                try {
                    $connection->insert('sogan_commerce_sku_sequence')
                        ->fields(['prefix' => $prefix, 'last' => $next])
                        ->execute();
                } catch (\Exception $e) {
                    // Insert failed due to race/duplicate; retry selecting and incrementing.
                    $query2 = $connection->query("SELECT last FROM {sogan_commerce_sku_sequence} WHERE prefix = :prefix FOR UPDATE", [':prefix' => $prefix]);
                    $row2 = $query2->fetchAssoc();
                    if ($row2) {
                        $next = (int) $row2['last'] + 1;
                        $connection->update('sogan_commerce_sku_sequence')
                            ->fields(['last' => $next])
                            ->condition('prefix', $prefix)
                            ->execute();
                    }
                }
            }
            // If we reach here, let the transaction object be destructed to
            // commit the transaction implicitly; no explicit commit method is
            // available on the transaction object in Drupal DB API.
            return $next;
        } catch (\Throwable $e) {
            // Rollback happens automatically when the transaction object
            // is destructed upon exit if not committed, but explicitly throw
            // the exception here to surface it.
            throw $e;
        }
    }
}
