<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use GuzzleHttp\ClientInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\File\FileExists;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\Core\Database\Connection;

/**
 * Service to create Media entities from a remote URL.
 *
 * It checks existing media by the module-specific `field_remote_url` (uri)
 * before attempting to download or create a new media entity.
 */
class RemoteMediaManager
{
    protected $entityTypeManager;
    protected $fileSystem;
    protected $streamWrapperManager;
    protected $httpClient;
    protected $fileRepository;
    protected $fileUsage;
    protected $entityFieldManager;
    protected $transliteration;
    protected $logger;
    protected $configFactory;
    protected $database;
    protected $imageFactory;

    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        FileSystemInterface $file_system,
        StreamWrapperManagerInterface $stream_wrapper_manager,
        ClientInterface $http_client,
        FileRepositoryInterface $file_repository,
        FileUsageInterface $file_usage,
        EntityFieldManagerInterface $entity_field_manager,
        TransliterationInterface $transliteration,
        LoggerChannelFactoryInterface $logger_factory,
        ConfigFactoryInterface $config_factory,
        Connection $database,
        $image_factory = NULL
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->fileSystem = $file_system;
        $this->streamWrapperManager = $stream_wrapper_manager;
        $this->httpClient = $http_client;
        $this->fileRepository = $file_repository;
        $this->fileUsage = $file_usage;
        $this->entityFieldManager = $entity_field_manager;
        $this->transliteration = $transliteration;
        $this->logger = $logger_factory->get('sogan_commerce_product');
        $this->configFactory = $config_factory;
        $this->database = $database;
        $this->imageFactory = $image_factory;
    }

    /**
     * Create or return an existing Media entity from remote URL.
     *
     * @param string $url
     *   Fully qualified remote URL to download.
     * @param string|null $title
     *   Optional title for the Media entity; used for filename and 'name' field.
     * @param array $context
     *   Optional context array containing entity references for token replacement.
     *   Keys: 'supplier_node', 'product', 'variation'.
     *
     * @return int|null
     *   Media entity id or NULL on failure.
     */
    public function createMediaFromRemoteUrl(string $url, ?string $title = NULL, array $context = []): ?int
    {
        $url = trim($url);
        if (empty($url)) {
            return NULL;
        }

        // 1) Find existing media by field_remote_url.uri (avoid re-download) if
        //    the field storage actually exists.
        $ids = [];
        $storage = $this->entityTypeManager->getStorage('field_storage_config');
        $field_storage = $storage->load('media.field_remote_url');
        if ($field_storage) {
            $query = $this->entityTypeManager->getStorage('media')->getQuery()
                ->condition('field_remote_url.uri', $url)
                ->range(0, 1)
                // Access bypass intentional: Backend media lookup without user context.
                ->accessCheck(FALSE);
            $ids = $query->execute();
        }
        if (!empty($ids)) {
            return (int) reset($ids);
        }

        // 2) Download remote asset.
        try {
            $response = $this->httpClient->request('GET', $url, ['headers' => ['User-Agent' => 'sogan_commerce_product/1.0']]);
            $status = $response->getStatusCode();
            if ($status !== 200) {
                return NULL;
            }
            $data = (string) $response->getBody()->getContents();
            $content_type = $response->getHeaderLine('Content-Type');
        } catch (\Throwable $e) {
            $this->logger->warning('Failed to fetch remote URL: @url; @msg', ['@url' => $url, '@msg' => $e->getMessage()]);
            return NULL;
        }

        // 3) Determine filename and extension.
        $path_info = pathinfo(parse_url($url, PHP_URL_PATH) ?? '');
        $original_basename = $path_info['basename'] ?? 'remote_file';
        $extension = $path_info['extension'] ?? NULL;
        if (empty($extension) && $content_type) {
            // Map content type to extension for images only (common ones).
            $map = [
                'image/jpeg' => 'jpg',
                'image/png' => 'png',
                'image/gif' => 'gif',
                'image/webp' => 'webp',
            ];
            $extension = $map[$content_type] ?? NULL;
        }

        // Get filename pattern from configuration and replace tokens
        $config = $this->configFactory->get('sogan_commerce_product.sku_settings');
        $filename_pattern = $config->get('media_filename_pattern') ?: '[title]';

        // Replace tokens in the filename pattern with context
        $base_filename = $this->replaceTokens($filename_pattern, $title ?: $original_basename, $url, $context);

        // Fallback to original basename if pattern results in empty string
        if (empty($base_filename)) {
            $base_filename = $original_basename;
        }

        // Apply sanitization using Drupal's file system service
        $file_system_service = $this->fileSystem;
        // Use Drupal's basename method which respects /admin/config/media/file-system settings
        $base_filename = $file_system_service->basename($base_filename);

        // Transliterate Turkish and other non-ASCII characters to English equivalents for better SEO
        // This converts: ı->i, ş->s, ğ->g, ü->u, ö->o, ç->c, etc.
        $base_filename = $this->transliteration->transliterate($base_filename, 'tr', '_');

        // Additional sanitization: replace spaces with hyphens, lowercase, remove special chars
        $base_filename = strtolower($base_filename);
        $base_filename = preg_replace('/\s+/', '-', $base_filename); // Replace spaces with hyphens
        $base_filename = preg_replace('/[^a-z0-9\-_.]/', '', $base_filename); // Remove special chars
        $base_filename = preg_replace('/-+/', '-', $base_filename); // Replace multiple hyphens with single
        $base_filename = trim($base_filename, '-'); // Remove leading/trailing hyphens

        // Ensure extension is present
        if ($extension && !str_ends_with(strtolower($base_filename), '.' . strtolower($extension))) {
            // Remove any existing extension first
            $base_filename = preg_replace('/\.[^.]+$/', '', $base_filename);
            $filename = $base_filename . '.' . $extension;
        } else {
            $filename = $base_filename;
        }

        // Get directory pattern from configuration and replace tokens
        $config = $this->configFactory->get('sogan_commerce_product.sku_settings');
        $directory_pattern = $config->get('media_directory_pattern') ?: 'public://product_media';
        $directory = $this->replaceTokens($directory_pattern, $title, $url, $context);

        // Ensure destination directory exists
        $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

        // Manually ensure unique filename to prevent overwriting
        $destination = $directory . '/' . $filename;
        $counter = 0;
        $path_info = pathinfo($filename);
        $name = $path_info['filename'];
        $ext = $path_info['extension'] ?? '';
        $ext_str = $ext ? '.' . $ext : '';

        // Clear stat cache to ensure file_exists sees recently created files
        clearstatcache(TRUE, $destination);

        while (file_exists($destination) || $this->fileUriExistsInDb($destination)) {
            $counter++;
            $new_filename = $name . '_' . $counter . $ext_str;
            $destination = $directory . '/' . $new_filename;
            // Clear cache for the new destination candidate
            clearstatcache(TRUE, $destination);
        }

        // 4) Save file to public:// (rename if exists)
        try {
            // We already ensured uniqueness, but keep Rename as a safety fallback
            $file = $this->fileRepository->writeData($data, $destination, FileExists::Rename);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to save remote file: @msg', ['@msg' => $e->getMessage()]);
            return NULL;
        }
        if (!$file instanceof File) {
            return NULL;
        }

        // 4.5) Convert image format if enabled
        $should_convert = (bool) ($config->get('convert_formats') ?? FALSE);
        if ($should_convert) {
            $target_format = $config->get('target_format') ?? 'jpeg';
            $converted_file = $this->convertImageFormat($file, $target_format);
            if ($converted_file) {
                // Delete original file and use converted one
                try {
                    $file->delete();
                } catch (\Throwable $e) {
                    // Ignore deletion error
                }
                $file = $converted_file;
            }
        }

        // 5) Set file permanent & save.
        $file->setPermanent();
        $file->save();

        // 6) Determine media bundle and file image field name.
        $media_bundle = 'image';
        // Find first image field on the image bundle.
        $bundle_definitions = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
        if (!isset($bundle_definitions['image'])) {
            // Fall back to first available media bundle that has an image field.
            foreach ($bundle_definitions as $bdef) {
                $bundle_id = $bdef->id();
                $fields = $this->entityFieldManager->getFieldDefinitions('media', $bundle_id);
                foreach ($fields as $field) {
                    if ($field->getType() === 'image') {
                        $media_bundle = $bundle_id;
                        break 2;
                    }
                }
            }
        }

        // Find image field for the bundle.
        // Prefer the canonical `field_media_image` if it exists, otherwise use
        // the first available field of type 'image'. This avoids writing to a
        // base 'thumbnail' field or other UI-only field which may be overridden
        // by Media internals or base fields.
        $image_field_name = NULL;
        $fields = $this->entityFieldManager->getFieldDefinitions('media', $media_bundle);
        if (isset($fields['field_media_image']) && $fields['field_media_image']->getType() === 'image') {
            $image_field_name = 'field_media_image';
        } else {
            foreach ($fields as $field_name => $field_def) {
                if ($field_def->getType() === 'image') {
                    $image_field_name = $field_name;
                    break;
                }
            }
        }
        if (!$image_field_name) {
            $this->logger->error('No image/file field found for media bundle @bundle; cannot create media from remote url: @url', ['@bundle' => $media_bundle, '@url' => $url]);
            // Remove file to avoid orphan files.
            try {
                $file->delete();
            } catch (\Throwable $e) {
            }
            return NULL;
        }

        // 7) Create the media entity.
        $name = $title ?: $file->getFilename();
        $media_storage = $this->entityTypeManager->getStorage('media');

        $media_data = [
            'bundle' => $media_bundle,
            'name' => $name,
            $image_field_name => [
                'target_id' => $file->id(),
                'alt' => $name,
                'title' => $name,
            ],
        ];

        // Only add field_remote_url if it exists on this bundle
        if ($this->entityFieldManager->getFieldDefinitions('media', $media_bundle)['field_remote_url'] ?? FALSE) {
            $media_data['field_remote_url'] = [
                'uri' => $url,
            ];
        }

        $media = $media_storage->create($media_data);
        try {
            $media->save();
        } catch (\Throwable $e) {
            $this->logger->error('Failed to create media: @msg', ['@msg' => $e->getMessage()]);
            // Cleanup: remove file.
            $file->delete();
            return NULL;
        }

        // Register file usage.
        try {
            $this->fileUsage->add($file, 'sogan_commerce_product', 'media', $media->id());
        } catch (\Throwable $e) {
            // Ignore usage addition failure.
        }

        return $media->id();
    }

    /**
     * Check if a file URI exists in the database.
     * 
     * @param string $uri
     *   The URI to check.
     * 
     * @return bool
     *   TRUE if the URI exists in file_managed table, FALSE otherwise.
     */
    protected function fileUriExistsInDb(string $uri): bool
    {
        $query = $this->database->select('file_managed', 'f');
        $query->addExpression('COUNT(*)');
        $query->condition('uri', $uri);
        return (bool) $query->execute()->fetchField();
    }


    protected function sanitizedFileName(string $input)
    {
        $result = (string) $this->transliteration->transliterate($input, 'en', '_');
        // Replace non-alphanumeric characters with underscores.
        $result = preg_replace('/[^A-Za-z0-9\-_\.]/', '_', $result);
        $result = preg_replace('#_+#', '_', $result);
        $result = trim($result, '_');
        if (empty($result)) {
            $result = 'remote_file';
        }
        return $result;
    }

    /**
     * Convert an image file to a different format.
     *
     * @param \Drupal\file\Entity\File $file
     *   The source file entity.
     * @param string $target_format
     *   Target format: jpeg, png, webp, or avif.
     *
     * @return \Drupal\file\Entity\File|null
     *   New converted file or NULL on failure.
     */
    protected function convertImageFormat(File $file, string $target_format): ?File
    {
        $valid_formats = ['jpeg', 'jpg', 'png', 'webp', 'avif'];
        if (!in_array($target_format, $valid_formats)) {
            return NULL;
        }

        // Normalize jpeg/jpg
        if ($target_format === 'jpg') {
            $target_format = 'jpeg';
        }

        $source_uri = $file->getFileUri();
        $source_path = $this->fileSystem->realpath($source_uri);

        if (!$source_path || !file_exists($source_path)) {
            return NULL;
        }

        // Use Drupal's Image toolkit
        $image = $this->imageFactory->get($source_uri);
        if (!$image->isValid()) {
            $this->logger->warning('Invalid image file: @uri', ['@uri' => $source_uri]);
            return NULL;
        }

        // Get current extension and check if conversion is needed
        $current_mime = $image->getMimeType();
        $current_format = $this->mimeToFormat($current_mime);

        if ($current_format === $target_format) {
            // Already in target format
            return NULL;
        }

        // Build new filename with new extension
        $original_filename = $file->getFilename();
        $pathinfo = pathinfo($original_filename);
        $base_name = $pathinfo['filename'] ?? 'converted';
        $new_filename = $base_name . '.' . $target_format;

        $directory = dirname($source_uri);
        $new_uri = $directory . '/' . $new_filename;

        // Ensure uniqueness for the converted file
        $counter = 0;
        // Clear stat cache for the new URI candidate
        clearstatcache(TRUE, $new_uri);

        while (file_exists($new_uri) || $this->fileUriExistsInDb($new_uri)) {
            $counter++;
            $new_filename = $base_name . '_' . $counter . '.' . $target_format;
            $new_uri = $directory . '/' . $new_filename;
            clearstatcache(TRUE, $new_uri);
        }

        // Convert the image
        try {
            // For WebP and AVIF, we need to ensure the toolkit supports them
            $toolkit = $image->getToolkitId();

            if (in_array($target_format, ['webp', 'avif']) && $toolkit !== 'gd') {
                $this->logger->warning('Format @format may not be supported by toolkit @toolkit', [
                    '@format' => $target_format,
                    '@toolkit' => $toolkit,
                ]);
            }

            // Apply conversion by saving with new extension
            $success = $image->save($new_uri);

            if (!$success) {
                $this->logger->error('Failed to convert image to @format', ['@format' => $target_format]);
                return NULL;
            }

            // Create new file entity
            $new_file = File::create([
                'uri' => $new_uri,
                'filename' => $new_filename,
                'filemime' => $this->formatToMime($target_format),
                'status' => 0, // Temporary initially
            ]);
            $new_file->save();

            return $new_file;
        } catch (\Throwable $e) {
            $this->logger->error('Image conversion failed: @msg', ['@msg' => $e->getMessage()]);
            return NULL;
        }
    }

    /**
     * Convert MIME type to format name.
     */
    protected function mimeToFormat(string $mime): string
    {
        $map = [
            'image/jpeg' => 'jpeg',
            'image/jpg' => 'jpeg',
            'image/png' => 'png',
            'image/webp' => 'webp',
            'image/avif' => 'avif',
            'image/gif' => 'gif',
        ];
        return $map[$mime] ?? 'jpeg';
    }

    /**
     * Convert format name to MIME type.
     */
    protected function formatToMime(string $format): string
    {
        $map = [
            'jpeg' => 'image/jpeg',
            'jpg' => 'image/jpeg',
            'png' => 'image/png',
            'webp' => 'image/webp',
            'avif' => 'image/avif',
            'gif' => 'image/gif',
        ];
        return $map[$format] ?? 'image/jpeg';
    }

    /**
     * Replace tokens in a pattern string.
     *
     * Supported tokens:
     * - [title] - Media/product title (sanitized)
     * - [date:year] - Current year (e.g., 2025)
     * - [date:month] - Current month with leading zero (e.g., 01, 12)
     * - [date:day] - Current day with leading zero (e.g., 01, 31)
     * - [random:hash] - Random 8-character hash for uniqueness
     * - [supplier:field_name] - Supplier node field value
     * - [product:field_name] - Product field value
     *
     * @param string $pattern
     *   The pattern string containing tokens.
     * @param string|null $title
     *   Optional title for [title] token replacement.
     * @param string|null $url
     *   Optional URL for context.
     * @param array $context
     *   Optional context array with entity references:
     *   - 'supplier_node': Supplier product node entity
     *   - 'product': Commerce product entity
     *   - 'variation': Commerce product variation entity
     *
     * @return string
     *   The pattern with tokens replaced.
     */
    protected function replaceTokens(string $pattern, ?string $title = NULL, ?string $url = NULL, array $context = []): string
    {
        $replacements = [];

        // [title] - Transliterated and sanitized title
        if (strpos($pattern, '[title]') !== FALSE) {
            $safe_title = $title ?: 'media';
            // Transliterate first for Turkish characters
            $safe_title = $this->transliteration->transliterate($safe_title, 'tr', '_');
            // Sanitize: lowercase, replace spaces with hyphens, remove special chars
            $safe_title = strtolower($safe_title);
            $safe_title = preg_replace('/\s+/', '-', $safe_title);
            $safe_title = preg_replace('/[^a-z0-9\-_]/', '', $safe_title);
            $safe_title = preg_replace('/-+/', '-', $safe_title);
            $safe_title = trim($safe_title, '-');
            $replacements['[title]'] = $safe_title ?: 'media';
        }

        // Date tokens
        if (strpos($pattern, '[date:') !== FALSE) {
            $replacements['[date:year]'] = date('Y');
            $replacements['[date:month]'] = date('m');
            $replacements['[date:day]'] = date('d');
        }

        // Random hash token
        if (strpos($pattern, '[random:hash]') !== FALSE) {
            $replacements['[random:hash]'] = substr(md5(uniqid((string) mt_rand(), TRUE)), 0, 8);
        }

        // Entity field tokens - [supplier:field_name] and [product:field_name]
        if (strpos($pattern, '[supplier:') !== FALSE || strpos($pattern, '[product:') !== FALSE) {
            // Extract all entity field tokens from pattern
            preg_match_all('/\[(supplier|product):([a-z0-9_]+)\]/', $pattern, $matches, PREG_SET_ORDER);

            foreach ($matches as $match) {
                $entity_type = $match[1]; // 'supplier' or 'product'
                $field_name = $match[2];
                $token = $match[0]; // Full token like [supplier:field_sku]

                $entity = NULL;
                if ($entity_type === 'supplier' && isset($context['supplier_node'])) {
                    $entity = $context['supplier_node'];
                } elseif ($entity_type === 'product' && isset($context['product'])) {
                    $entity = $context['product'];
                }

                if ($entity && $entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
                    $field_value = $this->extractFieldValue($entity, $field_name);
                    if ($field_value !== NULL) {
                        $replacements[$token] = $field_value;
                    }
                }
            }
        }

        // Apply replacements
        return str_replace(array_keys($replacements), array_values($replacements), $pattern);
    }

    /**
     * Extract and sanitize a field value from an entity for use in file paths.
     *
     * @param mixed $entity
     *   The entity to extract the field from.
     * @param string $field_name
     *   The field name.
     *
     * @return string|null
     *   Sanitized field value or NULL if not extractable.
     */
    protected function extractFieldValue($entity, string $field_name): ?string
    {
        try {
            $field = $entity->get($field_name);
            $value = NULL;

            // Handle different field types
            $field_type = $field->getFieldDefinition()->getType();

            switch ($field_type) {
                case 'string':
                case 'string_long':
                case 'text':
                case 'text_long':
                case 'text_with_summary':
                    $value = $field->value;
                    break;

                case 'integer':
                case 'decimal':
                case 'float':
                    $value = (string) $field->value;
                    break;

                case 'entity_reference':
                    // For entity references, use the referenced entity's label
                    $referenced = $field->entity;
                    if ($referenced) {
                        $value = $referenced->label();
                    }
                    break;

                case 'list_string':
                case 'list_integer':
                    $value = $field->value;
                    break;

                default:
                    // For other types, try to get the value
                    if (isset($field->value)) {
                        $value = $field->value;
                    }
            }

            if ($value === NULL || $value === '') {
                return NULL;
            }

            // Sanitize the value for file paths
            $sanitized = $this->transliteration->transliterate((string) $value, 'tr', '_');
            $sanitized = strtolower($sanitized);
            $sanitized = preg_replace('/\s+/', '-', $sanitized);
            $sanitized = preg_replace('/[^a-z0-9\-_]/', '', $sanitized);
            $sanitized = preg_replace('/-+/', '-', $sanitized);
            $sanitized = trim($sanitized, '-');

            return $sanitized ?: NULL;
        } catch (\Throwable $e) {
            $this->logger->warning('Failed to extract field @field: @msg', [
                '@field' => $field_name,
                '@msg' => $e->getMessage(),
            ]);
            return NULL;
        }
    }
}
