<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Service to build product tables (supplier products, variations).
 */
class ProductTableBuilder
{

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The supplier product field resolver.
   *
   * @var \Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver
   */
  protected $supplierFieldResolver;

  /**
   * The commerce stock service manager.
   *
   * @var \Drupal\commerce_stock\StockServiceManagerInterface
   */
  protected $stockServiceManager;

  /**
   * Constructs a new ProductTableBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver $supplier_field_resolver
   *   The supplier product field resolver.
   * @param \Drupal\commerce_stock\StockServiceManagerInterface $stock_service_manager
   *   The stock service manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, SupplierProductFieldResolver $supplier_field_resolver, $stock_service_manager)
  {
    $this->entityTypeManager = $entity_type_manager;
    $this->supplierFieldResolver = $supplier_field_resolver;
    $this->stockServiceManager = $stock_service_manager;
  }

  /**
   * Build the supplier products table.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   * @param array $options
   *   Display options (column visibility, labels, etc.).
   *
   * @return array
   *   Render array for the table.
   */
  public function buildSupplierTable(ProductInterface $product, array $options = [])
  {
    $product_id = $product->id();

    // Default options
    $default_options = [
      'show_product_name' => TRUE,
      'show_supplier' => TRUE,
      'show_sku' => TRUE,
      'show_cost_price' => TRUE,
      'show_list_price' => TRUE,
      'show_suggested_price' => TRUE,
      'show_stock' => TRUE,
      'show_ai_status' => TRUE,
      'label_product_name' => 'Product Name',
      'label_supplier' => 'Supplier',
      'label_sku' => 'SKU',
      'label_cost_price' => 'Cost Price',
      'label_list_price' => 'List Price',
      'label_suggested_price' => 'Suggested Price',
      'label_stock' => 'Stock',
      'label_ai_status' => 'AI Status',
      'show_column_labels' => TRUE,
      'link_sku' => TRUE,
      'show_only_instock' => FALSE,
      'compact_table' => FALSE,
      'table_class' => 'views-table',
    ];
    $options = array_merge($default_options, $options);

    // Get all required data
    $supplier_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier');
    $sku_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_sku');
    $url_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_url');
    $cost_price_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_cost_price');
    $list_price_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_list_price');
    $suggested_price_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_suggested_price');
    $stock_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_stock');

    // Combine all data
    $rows = [];
    $seen_combinations = [];

    foreach ($supplier_data as $variation_key => $variation_info) {
      foreach ($variation_info['supplier_fields'] as $index => $supplier_field) {
        // Get supplier name
        $supplier_name = '';
        if (!empty($supplier_field['field_value']) && is_array($supplier_field['field_value'])) {
          $supplier_tid = $supplier_field['field_value'][0]['target_id'] ?? NULL;
          if ($supplier_tid) {
            $term = $this->entityTypeManager->getStorage('taxonomy_term')->load($supplier_tid);
            if ($term) {
              $supplier_name = $term->label();
            }
          }
        }

        // Get SKU
        $sku = $sku_data[$variation_key]['supplier_fields'][$index]['field_value'] ?? '';

        // Skip if no supplier name or SKU
        if (empty($supplier_name) || empty($sku)) {
          continue;
        }

        // Check for duplicates
        $unique_key = $supplier_name . '|||' . $sku;
        if (isset($seen_combinations[$unique_key])) {
          continue;
        }
        $seen_combinations[$unique_key] = TRUE;

        // Get URL
        $url = '';
        if (isset($url_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $url_value = $url_data[$variation_key]['supplier_fields'][$index]['field_value'];
          if (is_array($url_value) && isset($url_value[0]['uri'])) {
            $url = $url_value[0]['uri'];
          } elseif (is_string($url_value)) {
            $url = $url_value;
          }
        }

        // Get Cost Price
        $cost_price = '';
        if (isset($cost_price_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $cost_price = $this->formatPrice($cost_price_data[$variation_key]['supplier_fields'][$index]['field_value']);
        }

        // Get List Price
        $list_price = '';
        if (isset($list_price_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $list_price = $this->formatPrice($list_price_data[$variation_key]['supplier_fields'][$index]['field_value']);
        }

        // Get Suggested Price
        $suggested_price = '';
        if (isset($suggested_price_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $suggested_price = $this->formatPrice($suggested_price_data[$variation_key]['supplier_fields'][$index]['field_value']);
        }

        // Get Stock
        $stock = '';
        if (isset($stock_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $stock = $stock_data[$variation_key]['supplier_fields'][$index]['field_value'];
        }

        // Filter out-of-stock items if option is enabled
        if ($options['show_only_instock']) {
          if ($stock === '' || $stock === NULL || (is_numeric($stock) && (float)$stock <= 0)) {
            continue;
          }
        }

        // Get product name from supplier product node
        $product_name = $supplier_field['supplier_product_title'] ?? '';
        $supplier_product_id = $supplier_field['supplier_product_id'];

        // Get AI enhancement status
        $ai_status = [];
        if ($options['show_ai_status'] && $supplier_product_id) {
          $ai_status = $this->getAiEnhancementStatus($supplier_product_id);
        }

        $rows[] = [
          'product_name' => $product_name,
          'supplier_product_id' => $supplier_product_id,
          'supplier' => $supplier_name,
          'sku' => $sku,
          'url' => $url,
          'cost_price' => $cost_price,
          'list_price' => $list_price,
          'suggested_price' => $suggested_price,
          'stock' => $stock,
          'ai_status' => $ai_status,
        ];
      }
    }

    if (empty($rows)) {
      return ['#markup' => $this->t('No supplier products found.')];
    }

    $header = [];
    $table_rows = [];
    $show_remove = count($rows) > 1; // Only show remove button if more than 1 supplier product

    // Build header based on visible columns
    if ($options['show_column_labels']) {
      if ($options['show_product_name']) {
        $header[] = $options['label_product_name'];
      }
      if ($options['show_supplier']) {
        $header[] = $options['label_supplier'];
      }
      if ($options['show_sku']) {
        $header[] = $options['label_sku'];
      }
      if ($options['show_stock']) {
        $header[] = $options['label_stock'];
      }
      if ($options['show_cost_price']) {
        $header[] = $options['label_cost_price'];
      }
      if ($options['show_list_price']) {
        $header[] = $options['label_list_price'];
      }
      if ($options['show_suggested_price']) {
        $header[] = $options['label_suggested_price'];
      }
      if ($options['show_ai_status']) {
        $header[] = $options['label_ai_status'];
      }
      if ($show_remove) {
        $header[] = '';
      }
    }

    // Build rows
    foreach ($rows as $row_data) {
      $row = [];

      if ($options['show_product_name']) {
        $product_name_link = [
          '#type' => 'link',
          '#title' => $row_data['product_name'],
          '#url' => Url::fromRoute('entity.node.canonical', ['node' => $row_data['supplier_product_id']]),
        ];
        $row[] = ['data' => $product_name_link];
      }

      if ($options['show_supplier']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['supplier'])]];
      }

      if ($options['show_sku']) {
        if ($options['link_sku'] && !empty($row_data['url'])) {
          $row[] = [
            'data' => [
              '#type' => 'link',
              '#title' => $row_data['sku'],
              '#url' => Url::fromUri($row_data['url']),
              '#attributes' => ['target' => '_blank'],
            ],
          ];
        } else {
          $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['sku'])]];
        }
      }

      if ($options['show_stock']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['stock'])]];
      }

      if ($options['show_cost_price']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['cost_price'])]];
      }

      if ($options['show_list_price']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['list_price'])]];
      }

      if ($options['show_suggested_price']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['suggested_price'])]];
      }

      if ($options['show_ai_status']) {
        $row[] = ['data' => $this->renderAiStatus($row_data['ai_status'] ?? [])];
      }

      if ($show_remove) {
        $remove_url = Url::fromRoute('sogan_commerce_product.remove_supplier', [
          'product_id' => $product_id,
          'supplier_product_id' => $row_data['supplier_product_id'],
        ]);
        $remove_button = [
          '#type' => 'link',
          '#title' => $this->t('Remove'),
          '#url' => $remove_url,
          '#attributes' => [
            'class' => ['button', 'button--small', 'button--danger', 'use-ajax'],
            'data-dialog-type' => 'modal',
            'title' => $this->t('Remove this supplier product'),
          ],
        ];
        $row[] = ['data' => $remove_button];
      }

      $table_rows[] = $row;
    }

    $table_classes = [$options['table_class']];
    if ($options['compact_table']) {
      $table_classes[] = 'supplier-products-table-compact';
    }

    return [
      '#theme' => 'table',
      '#header' => $options['show_column_labels'] ? $header : [],
      '#rows' => $table_rows,
      '#attributes' => [
        'class' => $table_classes,
      ],
      '#empty' => $this->t('No supplier products found.'),
      '#attached' => [
        'library' => ['sogan_commerce_product/supplier-table-compact', 'core/drupal.dialog.ajax'],
      ],
    ];
  }

  /**
   * Build the product variations table.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   * @param array $options
   *   Display options.
   *
   * @return array
   *   Render array for the table.
   */
  public function buildVariationTable(ProductInterface $product, array $options = [])
  {
    // Default options
    $default_options = [
      'show_variation_title' => TRUE,
      'show_sku' => TRUE,
      'show_price' => TRUE,
      'show_list_price' => TRUE,
      'show_cost_price' => TRUE,
      'show_stock' => TRUE,
      'show_logistics_code' => TRUE,
      'label_variation_title' => 'Variation',
      'label_sku' => 'SKU',
      'label_price' => 'Price',
      'label_list_price' => 'List Price',
      'label_cost_price' => 'Cost Price',
      'label_stock' => 'Stock',
      'label_logistics_code' => 'Logistics Code',
      'show_column_labels' => TRUE,
      'compact_table' => FALSE,
      'table_class' => 'views-table',
    ];
    $options = array_merge($default_options, $options);

    $variations = $product->getVariations();

    if (empty($variations)) {
      return ['#markup' => $this->t('No variations found.')];
    }

    $rows = [];

    // Load all locations for stock check
    $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
    $all_locations = $location_storage->loadMultiple();

    foreach ($variations as $variation) {
      $row = [];

      // Variation Title
      if ($options['show_variation_title']) {
        $row['variation_title'] = $variation->label();
      }

      // SKU
      if ($options['show_sku']) {
        $row['sku'] = $variation->getSku();
      }

      // Price
      if ($options['show_price']) {
        $price = $variation->getPrice();
        $row['price'] = $price ? number_format((float)$price->getNumber(), 2) . ' ' . $price->getCurrencyCode() : '';
      }

      // List Price
      if ($options['show_list_price']) {
        $list_price = $variation->getListPrice();
        $row['list_price'] = $list_price ? number_format((float)$list_price->getNumber(), 2) . ' ' . $list_price->getCurrencyCode() : '';
      }

      // Cost Price
      if ($options['show_cost_price']) {
        if ($variation->hasField('field_cost_price') && !$variation->get('field_cost_price')->isEmpty()) {
          $cost_price = $variation->get('field_cost_price')->first()->toPrice();
          $row['cost_price'] = $cost_price ? number_format((float)$cost_price->getNumber(), 2) . ' ' . $cost_price->getCurrencyCode() : '';
        } else {
          $row['cost_price'] = '';
        }
      }

      // Stock
      if ($options['show_stock']) {
        $stock_level = '';
        try {
          $stock_service = $this->stockServiceManager->getService($variation);
          $stock_checker = $stock_service->getStockChecker();
          $stock_level = $stock_checker->getTotalStockLevel($variation, $all_locations);
        } catch (\Exception $e) {
          // Fallback or ignore
        }
        $row['stock'] = $stock_level;
      }

      // Logistics Code
      if ($options['show_logistics_code']) {
        if ($variation->hasField('field_logistics_code') && !$variation->get('field_logistics_code')->isEmpty()) {
          $row['logistics_code'] = $variation->get('field_logistics_code')->value;
        } else {
          $row['logistics_code'] = '';
        }
      }

      $rows[] = $row;
    }

    $header = [];
    $table_rows = [];

    if ($options['show_column_labels']) {
      if ($options['show_variation_title']) {
        $header[] = $options['label_variation_title'];
      }
      if ($options['show_sku']) {
        $header[] = $options['label_sku'];
      }
      if ($options['show_price']) {
        $header[] = $options['label_price'];
      }
      if ($options['show_list_price']) {
        $header[] = $options['label_list_price'];
      }
      if ($options['show_cost_price']) {
        $header[] = $options['label_cost_price'];
      }
      if ($options['show_stock']) {
        $header[] = $options['label_stock'];
      }
      if ($options['show_logistics_code']) {
        $header[] = $options['label_logistics_code'];
      }
    }

    foreach ($rows as $row_data) {
      $row = [];

      if ($options['show_variation_title']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['variation_title'])]];
      }
      if ($options['show_sku']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['sku'])]];
      }
      if ($options['show_price']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['price'])]];
      }
      if ($options['show_list_price']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['list_price'])]];
      }
      if ($options['show_cost_price']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['cost_price'])]];
      }
      if ($options['show_stock']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['stock'])]];
      }
      if ($options['show_logistics_code']) {
        $row[] = ['data' => ['#markup' => htmlspecialchars($row_data['logistics_code'])]];
      }

      $table_rows[] = $row;
    }

    $table_classes = [$options['table_class']];
    if ($options['compact_table']) {
      $table_classes[] = 'product-variations-table-compact';
    }

    return [
      '#theme' => 'table',
      '#header' => $options['show_column_labels'] ? $header : [],
      '#rows' => $table_rows,
      '#attributes' => [
        'class' => $table_classes,
      ],
      '#empty' => $this->t('No variations found.'),
      '#attached' => [
        'library' => ['sogan_commerce_product/supplier-table-compact'],
      ],
    ];
  }

  /**
   * Format a price value.
   *
   * @param mixed $price_value
   *   The price field value.
   *
   * @return string
   *   Formatted price string.
   */
  protected function formatPrice($price_value)
  {
    if (!empty($price_value) && is_array($price_value)) {
      $number = $price_value[0]['number'] ?? NULL;
      $currency = $price_value[0]['currency_code'] ?? 'TRY';

      if ($number !== NULL) {
        return number_format((float)$number, 2) . ' ' . $currency;
      }
    }
    return '';
  }

  /**
   * Get the number of supplier products for a product.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return int
   *   The count of supplier products.
   */
  public function getSupplierProductCount(ProductInterface $product)
  {
    $product_id = $product->id();
    $supplier_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier');
    $sku_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_sku');

    $seen_combinations = [];
    $count = 0;

    foreach ($supplier_data as $variation_key => $variation_info) {
      foreach ($variation_info['supplier_fields'] as $index => $supplier_field) {
        // Get supplier name
        $supplier_name = '';
        if (!empty($supplier_field['field_value']) && is_array($supplier_field['field_value'])) {
          $supplier_tid = $supplier_field['field_value'][0]['target_id'] ?? NULL;
          if ($supplier_tid) {
            $term = $this->entityTypeManager->getStorage('taxonomy_term')->load($supplier_tid);
            if ($term) {
              $supplier_name = $term->label();
            }
          }
        }

        // Get SKU
        $sku = $sku_data[$variation_key]['supplier_fields'][$index]['field_value'] ?? '';

        // Skip if no supplier name or SKU
        if (empty($supplier_name) || empty($sku)) {
          continue;
        }

        // Check for duplicates
        $unique_key = $supplier_name . '|||' . $sku;
        if (isset($seen_combinations[$unique_key])) {
          continue;
        }
        $seen_combinations[$unique_key] = TRUE;
        $count++;
      }
    }

    return $count;
  }

  /**
   * Get the number of variations for a product.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return int
   *   The count of variations.
   */
  public function getVariationCount(ProductInterface $product)
  {
    $variation_ids = $product->getVariationIds();
    return count($variation_ids);
  }

  /**
   * Get AI enhancement status for a supplier product.
   *
   * @param int $supplier_product_id
   *   The supplier product node ID.
   *
   * @return array
   *   Array with keys for each AI field type and boolean values.
   */
  protected function getAiEnhancementStatus($supplier_product_id)
  {
    $status = [
      'title' => FALSE,
      'description' => FALSE,
      'attributes' => FALSE,
      'categories' => FALSE,
      'brand' => FALSE,
    ];

    try {
      $node = $this->entityTypeManager->getStorage('node')->load($supplier_product_id);
      if (!$node || $node->bundle() !== 'supplier_product') {
        return $status;
      }

      // Check each AI field
      if ($node->hasField('field_ai_title') && !$node->get('field_ai_title')->isEmpty()) {
        $status['title'] = TRUE;
      }
      if ($node->hasField('field_ai_description') && !$node->get('field_ai_description')->isEmpty()) {
        $status['description'] = TRUE;
      }
      if ($node->hasField('field_ai_attributes') && !$node->get('field_ai_attributes')->isEmpty()) {
        $status['attributes'] = TRUE;
      }
      if ($node->hasField('field_ai_suggested_categories') && !$node->get('field_ai_suggested_categories')->isEmpty()) {
        $status['categories'] = TRUE;
      }
      if ($node->hasField('field_ai_suggested_brand') && !$node->get('field_ai_suggested_brand')->isEmpty()) {
        $status['brand'] = TRUE;
      }
    } catch (\Exception $e) {
      // Return default status if node cannot be loaded.
    }

    return $status;
  }

  /**
   * Render AI enhancement status as HTML.
   *
   * @param array $status
   *   AI status array from getAiEnhancementStatus().
   *
   * @return array
   *   Render array for the AI status indicators.
   */
  protected function renderAiStatus(array $status)
  {
    $indicators = [
      'title' => $this->t('Title'),
      'description' => $this->t('Description'),
      'attributes' => $this->t('Attributes'),
      'categories' => $this->t('Categories'),
      'brand' => $this->t('Brand'),
    ];

    $boxes = [];
    foreach ($indicators as $key => $label) {
      $filled = !empty($status[$key]);
      $class = 'ai-status-box';
      if ($filled) {
        $class .= ' ai-status-box--enhanced';
        $title = $label . ': ' . $this->t('Enhanced');
      } else {
        $class .= ' ai-status-box--not-enhanced';
        $title = $label . ': ' . $this->t('Not Enhanced');
      }
      $boxes[] = '<span class="' . $class . '" title="' . $title . '"></span>';
    }

    return [
      '#type' => 'inline_template',
      '#template' => '<span class="ai-status-indicators">{{ boxes|raw }}</span>',
      '#context' => [
        'boxes' => implode('', $boxes),
      ],
      '#attached' => [
        'library' => ['supplier_products_ai_rewrite/ai-status'],
      ],
    ];
  }
}
