<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\sogan_commerce_product\Service\ProductBuilder;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\node\NodeInterface;

/**
 * Helper service to create Commerce Products and Variations from supplier nodes.
 */
class ProductCreator
{
    use StringTranslationTrait;

    protected EntityTypeManagerInterface $entityTypeManager;
    protected ProductBuilder $productBuilder;
    protected MessengerInterface $messenger;

    public function __construct(EntityTypeManagerInterface $entity_type_manager, ProductBuilder $product_builder, MessengerInterface $messenger)
    {
        $this->entityTypeManager = $entity_type_manager;
        $this->productBuilder = $product_builder;
        $this->messenger = $messenger;
    }

    /**
     * Create a commerce product and a default variation using a supplier node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node entity.
     *
     * @return \Drupal\commerce_product\Entity\ProductInterface|null
     *   The created product or NULL on failure.
     */
    public function createProductFromSupplier(NodeInterface $node)
    {
        if ($node->getEntityTypeId() !== 'node' || $node->bundle() !== 'supplier_product') {
            return NULL;
        }

        // Check if this supplier product is already linked to a product variation
        $existing_variation = $this->findExistingVariationForSupplier($node);
        if ($existing_variation) {
            $this->messenger->addWarning($this->t('Supplier product "@title" (ID: @nid) is already linked to variation @vid. Skipping creation.', [
                '@title' => $node->label(),
                '@nid' => $node->id(),
                '@vid' => $existing_variation->id(),
            ]));

            // Return the existing product instead of creating a new one
            $product = $existing_variation->getProduct();
            return $product;
        }

        // Delegate to the ProductBuilder service to ensure price/stock mapping.
        $product = $this->productBuilder->createProductFromSuppliers([$node]);
        return $product;

        // Create a variation first. Ensure a unique SKU by including node id.
        $sku = 'SP-' . $node->id() . '-' . time();
        $price = ['number' => '0.00', 'currency_code' => 'TRY'];
        $variation = $variation_storage->create([
            'type' => 'default',
            'sku' => $sku,
            'price' => $price,
        ]);
        $variation->save();

        // Create the product and attach store and variation.
        $stores = $store_storage->loadMultiple();
        $store_ids = [];
        if (!empty($stores)) {
            $store_ids[] = reset($stores)->id();
        }
        // Determine product title with AI fallback.
        $title = $node->label();
        if ($node->hasField('field_ai_title') && !$node->get('field_ai_title')->isEmpty()) {
            $title = (string) $node->get('field_ai_title')->value;
        } elseif ($node->hasField('field_original_product_title') && !$node->get('field_original_product_title')->isEmpty()) {
            $title = (string) $node->get('field_original_product_title')->value;
        }

        // Determine product body (description) with AI fallback.
        $description_value = NULL;
        $description_format = 'full_html';
        if ($node->hasField('field_ai_description') && !$node->get('field_ai_description')->isEmpty()) {
            $description_value = (string) $node->get('field_ai_description')->value;
            $description_format = $node->get('field_ai_description')->format ?? $description_format;
        } elseif ($node->hasField('field_supplier_product_descripti') && !$node->get('field_supplier_product_descripti')->isEmpty()) {
            $description_value = (string) $node->get('field_supplier_product_descripti')->value;
            // If original supplier desc is text with format, preserve it when possible.
            $description_format = $node->get('field_supplier_product_descripti')->format ?? $description_format;
        }

        $product = $product_storage->create([
            'type' => 'default',
            'title' => $title,
            'stores' => $store_ids,
        ]);
        // Set variations using API if available; otherwise set the field directly.
        if (method_exists($product, 'setVariations')) {
            $product->setVariations([$variation]);
        } else {
            $product->set('variations', [$variation]);
        }
        $product->save();

        // If we have a description, set the product body if available on product.
        if ($description_value !== NULL && $product->hasField('body')) {
            $product->set('body', [
                'value' => $description_value,
                'format' => $description_format,
            ]);
            $product->save();
        }

        return $product;
    }

    /**
     * Find an existing variation that references the given supplier product.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return \Drupal\commerce_product\Entity\ProductVariationInterface|null
     *   The existing variation or NULL if not found.
     */
    protected function findExistingVariationForSupplier(NodeInterface $node)
    {
        $variation_storage = $this->entityTypeManager->getStorage('commerce_product_variation');

        // Query for variations that reference this supplier product node
        $query = $variation_storage->getQuery()
            ->condition('field_source_supplier_products', $node->id())
            ->range(0, 1)
            ->accessCheck(FALSE);

        $ids = $query->execute();

        if (!empty($ids)) {
            $variation_id = reset($ids);
            return $variation_storage->load($variation_id);
        }

        return NULL;
    }
}
