<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;


use Drupal\sogan_commerce_product\Service\TaxonomyManager;

class ProductBuilder
{

    protected $entityTypeManager;
    protected $attributeManager;

    protected $remoteMediaManager;
    protected $taxonomyManager;
    protected $stockManager;
    protected $transactionManager;
    protected $imageFixQueue;
    /**
     * @var \Drupal\sogan_commerce_product\Service\PriceCalculator
     */
    protected $priceCalculator;
    /**
     * @var \Drupal\sogan_commerce_product\Service\LogisticsCodeGenerator
     */
    protected $logisticsCodeGenerator;
    /**
     * @var \Drupal\sogan_commerce_product\Service\AttributeResolver
     */
    protected $attributeResolver;
    /**
     * @var \Drupal\sogan_commerce_product\Service\ContentResolver
     */
    protected $contentResolver;
    /**
     * @var \Drupal\sogan_commerce_product\Service\VariationBuilder
     */
    protected $variationBuilder;
    /**
     * @var \Drupal\sogan_commerce_product\Service\StockSynchronizer
     */
    protected $stockSynchronizer;
    /**
     * @var \Drupal\sogan_commerce_product\Service\TaxRateResolver
     */
    protected $taxRateResolver;
    /**
     * @var \Drupal\sogan_commerce_product\Service\ImageHandler
     */
    protected $imageHandler;
    /**
     * @var \Drupal\Core\Config\ImmutableConfig
     */
    protected $config;
    /**
     * @var array
     */
    protected $mergePolicy;
    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;
    /**
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        AttributeManager $attribute_manager,

        RemoteMediaManager $remote_media_manager,
        TaxonomyManager $taxonomy_manager,
        ConfigFactoryInterface $config_factory,
        StockManager $stock_manager,
        TransactionManager $transaction_manager,
        ImageFixQueue $image_fix_queue,
        PriceCalculator $price_calculator,
        LogisticsCodeGenerator $logistics_code_generator,
        AttributeResolver $attribute_resolver,
        ContentResolver $content_resolver,
        VariationBuilder $variation_builder,
        StockSynchronizer $stock_synchronizer,
        TaxRateResolver $tax_rate_resolver,
        ImageHandler $image_handler,
        LoggerChannelFactoryInterface $logger_factory,
        MessengerInterface $messenger
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->attributeManager = $attribute_manager;

        $this->remoteMediaManager = $remote_media_manager;
        $this->taxonomyManager = $taxonomy_manager;
        $this->config = $config_factory->get('sogan_commerce_product.settings');
        $this->mergePolicy = $this->config->get('merge_policy') ?? [];
        $this->stockManager = $stock_manager;
        $this->transactionManager = $transaction_manager;
        $this->imageFixQueue = $image_fix_queue;
        $this->priceCalculator = $price_calculator;
        $this->logisticsCodeGenerator = $logistics_code_generator;
        $this->attributeResolver = $attribute_resolver;
        $this->contentResolver = $content_resolver;
        $this->variationBuilder = $variation_builder;
        $this->stockSynchronizer = $stock_synchronizer;
        $this->taxRateResolver = $tax_rate_resolver;
        $this->imageHandler = $image_handler;
        $this->logger = $logger_factory->get('sogan_commerce_product');
        $this->messenger = $messenger;
    }


    /**
     * Main Function: Creates a Store Product from Supplier Nodes.
     */
    public function createProductFromSuppliers(array $supplier_nodes)
    {
        if (empty($supplier_nodes)) return;

        // Wrap entire product creation in transaction for rollback capability
        return $this->transactionManager->executeInTransaction(function () use ($supplier_nodes) {
            return $this->executeProductCreation($supplier_nodes);
        }, 'product_creation');
    }

    /**
     * Internal method to execute product creation (called within transaction).
     */
    protected function executeProductCreation(array $supplier_nodes)
    {
        $all_nodes = $this->expandVariations($supplier_nodes);

        // 1. ANALYSIS: Determine which attributes are what
        $analysis = $this->attributeResolver->analyzeAttributes($all_nodes);

        // 2. CREATE MAIN PRODUCT

        // Use configured strategy for product name selection
        $title = $this->contentResolver->selectTextByStrategy($all_nodes, 'title', 'product_name_strategy');
        if (!$title) {
            // Fallback to first node label if strategy returns null
            $first_node = reset($all_nodes);
            $title = $first_node->label();
        }

        $product = Product::create([
            'type' => 'default',
            'title' => $title,
            'stores' => [1],
        ]);
        $product->save();

        // 2. IMAGES
        // Determine which nodes to process for images based on strategy
        $image_strategy = $this->config->get('merge_policy.product_image_strategy') ?? 'combine';
        $nodes_for_images = $all_nodes;

        if ($image_strategy === 'most_images') {
            $best_node = NULL;
            $max_count = -1;

            foreach ($all_nodes as $node) {
                $count = 0;
                // Count existing media items
                if ($node->hasField('field_images') && !$node->get('field_images')->isEmpty()) {
                    $count += count($node->get('field_images')->getValue());
                }
                // Count remote URLs (potential images)
                if ($node->hasField('field_supplier_image_urls') && !$node->get('field_supplier_image_urls')->isEmpty()) {
                    $count += count($node->get('field_supplier_image_urls')->getValue());
                }

                if ($count > $max_count) {
                    $max_count = $count;
                    $best_node = $node;
                }
            }

            if ($best_node) {
                $nodes_for_images = [$best_node];
            } else {
                $nodes_for_images = [];
            }
        }

        $media_items = [];
        $failed_image_urls = [];

        foreach ($nodes_for_images as $node) {
            // 2a. Existing Media
            if ($node->hasField('field_images') && !$node->get('field_images')->isEmpty()) {
                $items = $node->get('field_images')->getValue();
                foreach ($items as $item) {
                    $media_items[] = ['target_id' => $item['target_id']];
                }
            }

            // 2b. Remote Images (Download & Create Media)
            if ($node->hasField('field_supplier_image_urls') && !$node->get('field_supplier_image_urls')->isEmpty()) {
                $image_values = $node->get('field_supplier_image_urls')->getValue();

                foreach ($image_values as $item) {
                    $uri = $item['uri'] ?? ($item['value'] ?? NULL);
                    if (empty($uri)) {
                        continue;
                    }

                    // Pass supplier node as context for field token replacement
                    $context = [
                        'supplier_node' => $node,
                        'product' => $product,
                    ];

                    $media_id = $this->remoteMediaManager->createMediaFromRemoteUrl($uri, $title, $context);
                    if ($media_id) {
                        $media_items[] = ['target_id' => $media_id];
                    } else {
                        // Collect failed URLs for queueing
                        $failed_image_urls[] = $uri;
                        $this->logger->warning('Failed to create media from URL: @url for product @title (will retry via queue)', [
                            '@url' => $uri,
                            '@title' => $title,
                        ]);
                    }
                }
            }
        }

        // Deduplicate media items by target_id
        $unique_media_items = [];
        $seen_ids = [];
        foreach ($media_items as $item) {
            if (!isset($seen_ids[$item['target_id']])) {
                $unique_media_items[] = $item;
                $seen_ids[$item['target_id']] = TRUE;
            }
        }

        // Set all collected images on the product
        if (!empty($unique_media_items) && $product->hasField('field_product_images')) {
            $product->set('field_product_images', $unique_media_items);
            $product->save();
        }

        // Queue failed images for retry
        if (!empty($failed_image_urls)) {
            $this->imageFixQueue->addProduct($product, $failed_image_urls, 'creation');
        }

        // 3. ASSIGN PRODUCT ATTRIBUTES (Taxonomies)
        // We include Common Attributes (Case A) and Variation Attributes (Case B - Dual Record).
        // We DO NOT include Sparse Attributes (Case C) to avoid data pollution.

        $all_taxonomies = array_merge(
            $analysis['common_attributes'],
            $analysis['variation_attributes']
        );

        // Build a flat array of all taxonomy items to process at once.
        // This ensures that when fields are created and product is reloaded,
        // ALL taxonomies are re-processed together.
        $taxonomies_data = [];
        foreach ($all_taxonomies as $key => $values) {
            $values = (array) $values;
            foreach ($values as $val) {
                $taxonomies_data[] = ['name' => $key, 'value' => $val];
            }
        }

        // Process all taxonomies at once
        if (!empty($taxonomies_data)) {
            $product = $this->attributeManager->setProductTaxonomies($product, $taxonomies_data);
        }
        $product->save();

        // If we have a description, set the product body if available on product.
        // Use configured strategy for description selection
        $description_value = $this->contentResolver->selectTextByStrategy($all_nodes, 'description', 'product_description_strategy');

        if ($description_value !== NULL && $product->hasField('body')) {
            $product->set('body', [
                'value' => $description_value,
                'format' => 'full_html', // Default to full_html
            ]);
            $product->save();
        }

        // Map AI-suggested categories, brand and tags into the canonical
        // product fields used on this site.
        $first_node = reset($all_nodes);
        if ($first_node) {
            // Resolve and (if missing) auto-create taxonomy terms from AI-suggested
            // strings or term references. Use TaxonomyManager to return term
            // reference arrays suitable for setting on the product fields.
            if ($first_node->hasField('field_ai_suggested_categories') && !$first_node->get('field_ai_suggested_categories')->isEmpty()) {
                $raw = $first_node->get('field_ai_suggested_categories')->getValue();
                $resolved = $this->taxonomyManager->resolveSuggestedValues('product_category', $raw);
                if (!empty($resolved) && $product->hasField('field_product_category')) {
                    $product->set('field_product_category', $resolved);
                }
            }

            if ($first_node->hasField('field_ai_suggested_brand') && !$first_node->get('field_ai_suggested_brand')->isEmpty()) {
                $raw = $first_node->get('field_ai_suggested_brand')->getValue();
                $resolved = $this->taxonomyManager->resolveSuggestedValues('product_brand', $raw);
                if (!empty($resolved) && $product->hasField('field_product_brand')) {
                    $product->set('field_product_brand', $resolved);
                }
            }

            if ($first_node->hasField('field_ai_suggested_tags') && !$first_node->get('field_ai_suggested_tags')->isEmpty()) {
                $raw = $first_node->get('field_ai_suggested_tags')->getValue();
                $resolved = $this->taxonomyManager->resolveSuggestedValues('product_tag', $raw);
                if (!empty($resolved) && $product->hasField('field_product_tag')) {
                    $product->set('field_product_tag', $resolved);
                }
            }

            // Persist any taxonomy assignments made above.
            $product->save();
        }

        // 4. CREATE VARIATIONS
        // Filter out supplier nodes that are already linked to existing variations
        $nodes_to_process = [];
        $skipped_nodes = [];

        foreach ($all_nodes as $nid => $node) {
            $existing_variation = $this->findExistingVariationForSupplier($node);
            if ($existing_variation) {
                $skipped_nodes[$nid] = [
                    'node' => $node,
                    'variation' => $existing_variation,
                ];
            } else {
                $nodes_to_process[$nid] = $node;
            }
        }

        // Warn about skipped nodes
        if (!empty($skipped_nodes)) {
            foreach ($skipped_nodes as $nid => $info) {
                $this->messenger->addWarning(t('Supplier product "@title" (ID: @nid) is already linked to variation @vid. Skipping creation of duplicate variation.', [
                    '@title' => $info['node']->label(),
                    '@nid' => $nid,
                    '@vid' => $info['variation']->id(),
                ]));
            }
        }

        // Create variations only for nodes that aren't already linked
        $variation_keys = array_keys($analysis['variation_attributes']);

        // If NO variation attributes exist, create ONE variation with ALL suppliers as sources
        if (empty($variation_keys)) {
            // Create single variation with all supplier nodes
            $variation = $this->variationBuilder->createVariation(reset($nodes_to_process), $product, []);

            // Add all other suppliers to this variation's field_source_supplier_products
            $all_supplier_ids = [];
            foreach ($nodes_to_process as $node) {
                $all_supplier_ids[] = $node->id();
            }
            $variation->set('field_source_supplier_products', $all_supplier_ids);
            $variation->save();

            // Ensure aggregated stock is set for this variation across all
            // supplier nodes that were attached to it. Previously we only
            // created a transaction for the single node used to create the
            // variation; for the "no-variation-attributes" case we must
            // apply the grouped aggregation so the location reflects the
            // total from all supplier_product sources (e.g., 34 + 0 + 7 = 41).
            $this->stockSynchronizer->setVariationStocksGroupedBySupplier($variation, $all_nodes);

            // Set back-reference on all supplier nodes
            foreach ($nodes_to_process as $node) {
                if ($node->hasField('field_associated_product_variati')) {
                    $node->set('field_associated_product_variati', $variation->id());
                    $node->save();
                }
            }

            $product->addVariation($variation);
        } else {
            // Group suppliers by their attribute signatures, then create
            // one variation per group with all matching suppliers as sources.
            $groups = $this->attributeResolver->groupSuppliersByAttributes($nodes_to_process);

            foreach ($groups as $signature => $group) {
                // Use first node in group to create the variation
                $first_node = reset($group['nodes']);
                $variation = $this->variationBuilder->createVariation($first_node, $product, $variation_keys);

                // Add all nodes in group as sources
                $source_ids = array_map(function ($n) {
                    return $n->id();
                }, $group['nodes']);
                $variation->set('field_source_supplier_products', $source_ids);
                $variation->save();

                // Set aggregated stock from all sources
                $this->stockSynchronizer->setVariationStocksGroupedBySupplier($variation, $group['nodes']);

                // Set back-reference on all source nodes
                foreach ($group['nodes'] as $node) {
                    if ($node->hasField('field_associated_product_variati')) {
                        $node->set('field_associated_product_variati', $variation->id());
                        $node->save();
                    }
                }

                $product->addVariation($variation);
            }
        }

        // Generate Product Logistics Code
        $this->setProductLogisticsCodeFromSupplierNodes($product, $all_nodes);

        $product->save();

        // Show success message with details
        $total_suppliers = count($all_nodes);
        $created_variations = count($nodes_to_process);
        $skipped_variations = count($skipped_nodes);

        if ($created_variations > 0) {
            $this->messenger->addStatus(t('@product_name (NID: @nid) created with @variations variation(s) from @suppliers supplier product(s).', [
                '@product_name' => $product->getTitle(),
                '@nid' => $product->id(),
                '@variations' => $created_variations,
                '@suppliers' => $total_suppliers,
            ]));
        }

        if ($skipped_variations > 0 && $created_variations === 0) {
            $this->messenger->addWarning(t('Product "@product_name" (NID: @nid) was created but no new variations were added. All @count supplier product(s) were already linked to existing variations.', [
                '@product_name' => $product->getTitle(),
                '@nid' => $product->id(),
                '@count' => $skipped_variations,
            ]));
        }

        return $product;
    }

    /**
     * Find an existing variation that references the given supplier product.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return \Drupal\commerce_product\Entity\ProductVariationInterface|null
     *   The existing variation or NULL if not found.
     */
    protected function findExistingVariationForSupplier($node)
    {
        if (!$node instanceof \Drupal\node\NodeInterface) {
            return NULL;
        }

        $variation_storage = $this->entityTypeManager->getStorage('commerce_product_variation');

        // Query for variations that reference this supplier product node
        $query = $variation_storage->getQuery()
            ->condition('field_source_supplier_products', $node->id())
            ->range(0, 1)
            // Access bypass intentional: Backend service operation without user context.
            ->accessCheck(FALSE);

        $ids = $query->execute();

        if (!empty($ids)) {
            $variation_id = reset($ids);
            return $variation_storage->load($variation_id);
        }

        return NULL;
    }

    /**
     * Helper: Expands selected nodes to include their referenced variations.
     */
    protected function expandVariations(array $nodes)
    {
        $expanded = [];
        $query_criteria = [];
        $supplier_ids = [];
        $parent_skus = [];

        foreach ($nodes as $node) {
            $expanded[$node->id()] = $node;

            // 1. Try pre-computed field (fastest)
            if ($node->hasField('field_other_variation_products') && !$node->get('field_other_variation_products')->isEmpty()) {
                $referenced_entities = $node->get('field_other_variation_products')->referencedEntities();
                foreach ($referenced_entities as $ref_node) {
                    $expanded[$ref_node->id()] = $ref_node;
                }
            }
            // 2. Prepare for batch fallback query
            else {
                $supplier_id = $node->hasField('field_supplier') && !$node->get('field_supplier')->isEmpty()
                    ? $node->get('field_supplier')->target_id
                    : NULL;

                $parent_sku = $node->hasField('field_parent_sku') && !$node->get('field_parent_sku')->isEmpty()
                    ? $node->get('field_parent_sku')->value
                    : NULL;

                if ($supplier_id && $parent_sku) {
                    $query_criteria[] = [
                        'supplier' => $supplier_id,
                        'parent_sku' => $parent_sku,
                    ];
                    $supplier_ids[$supplier_id] = $supplier_id;
                    $parent_skus[$parent_sku] = $parent_sku;
                }
            }
        }

        // Execute batch fallback query if criteria collected
        if (!empty($query_criteria)) {
            $query = $this->entityTypeManager->getStorage('node')->getQuery()
                ->condition('type', 'supplier_product')
                ->condition('field_supplier', array_values($supplier_ids), 'IN')
                ->condition('field_parent_sku', array_values($parent_skus), 'IN')
                // Access bypass intentional: Backend service operation without user context.
                ->accessCheck(FALSE);

            $ids = $query->execute();
            if (!empty($ids)) {
                $check_nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($ids);
                foreach ($check_nodes as $check_node) {
                    // Filter in PHP to ensure exact pair matching
                    $check_supplier = $check_node->get('field_supplier')->target_id;
                    $check_parent = $check_node->get('field_parent_sku')->value;

                    foreach ($query_criteria as $criteria) {
                        if ($check_supplier == $criteria['supplier'] && $check_parent == $criteria['parent_sku']) {
                            // Match found - add to expanded list
                            $expanded[$check_node->id()] = $check_node;
                            break;
                        }
                    }
                }
            }
        }

        return $expanded;
    }



    /**
     * Set the logistics code on a variation using supplier term fields and
     * padded variation id.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariation $variation
     *   The variation entity to update.
     * @param \Drupal\taxonomy\TermInterface $supplier_term
     *   The supplier taxonomy term which contains supplier, city, and district codes.
     */
    /**
     * Set the logistics code on a variation using a single supplier term.
     * Backwards compatible wrapper for setVariationLogisticsCodeFromSupplierNodes.
     */
    /**
     * Backwards compatible wrapper: set logistics code using a supplier term
     * or supplier node. Delegates to setVariationLogisticsCodeFromSupplierNodes
     */
    public function setVariationLogisticsCode($variation, $supplier_term_or_node): void
    {
        if (!$variation->hasField('field_logistics_code')) {
            return;
        }
        // Re-use the multi-node setter for consistent behavior.
        $nodes = [];
        if ($supplier_term_or_node instanceof \Drupal\node\NodeInterface) {
            $nodes[] = $supplier_term_or_node;
        } elseif ($supplier_term_or_node instanceof \Drupal\taxonomy\TermInterface) {
            // Wrap a term into a synthetic node structure to allow reuse of the
            // FromSupplierNodes method which expects nodes.
            $fake_node = new \stdClass();
            $fake_node->field_supplier = (object) ['entity' => $supplier_term_or_node];
            $nodes[] = $fake_node;
        }
        $this->setVariationLogisticsCodeFromSupplierNodes($variation, $nodes);
        // The save is handled by the delegated method.
    }

    /**
     * Compute the logistics code body string from an array of supplier nodes.
     *
     * @param array $supplier_nodes
     *   Array of supplier product node entities.
     *
     * @return string
     *   The logistics code body (e.g., '34K' or '99XX' body part — city(2)+district(1)+supplier(1)).
     */
    public function computeLogisticsCodeBodyFromSupplierNodes(array $supplier_nodes): string
    {
        $city_codes = [];
        $district_codes = [];
        $supplier_codes = [];
        foreach ($supplier_nodes as $n) {
            $term = NULL;
            // Accept either a NodeInterface (entity) or a stdClass wrapper created
            // by the single-term wrapper function above.
            if ($n instanceof \Drupal\node\NodeInterface && $n->hasField('field_supplier') && !$n->get('field_supplier')->isEmpty()) {
                $term = $n->get('field_supplier')->entity;
            } elseif (is_object($n) && isset($n->field_supplier->entity)) {
                $term = $n->field_supplier->entity;
            }
            if (!$term) {
                continue;
            }
            if ($term->hasField('field_city_code') && !$term->get('field_city_code')->isEmpty()) {
                $city_codes[] = (string) $term->get('field_city_code')->value;
            }
            if ($term->hasField('field_district_code') && !$term->get('field_district_code')->isEmpty()) {
                $district_codes[] = (string) $term->get('field_district_code')->value;
            }
            if ($term->hasField('field_supplier_code') && !$term->get('field_supplier_code')->isEmpty()) {
                $supplier_codes[] = (string) $term->get('field_supplier_code')->value;
            }
        }

        // Deduplicate and filter empties
        $city_codes = array_values(array_unique(array_filter($city_codes)));
        $district_codes = array_values(array_unique(array_filter($district_codes)));
        $supplier_codes = array_values(array_unique(array_filter($supplier_codes)));

        // City: if >1 then '99', if 1 then value, else 'XX'
        if (count($city_codes) > 1) {
            $city = '99';
        } elseif (count($city_codes) === 1) {
            $city = $city_codes[0];
        } else {
            $city = 'XX';
        }
        // District: >1 -> 'X', 1 -> value, else 'X'
        if (count($district_codes) > 1) {
            $district = 'X';
        } elseif (count($district_codes) === 1) {
            $district = $district_codes[0];
        } else {
            $district = 'X';
        }
        // Supplier: >1 -> 'X', 1 -> value, else 'X'
        if (count($supplier_codes) > 1) {
            $supplier = 'X';
        } elseif (count($supplier_codes) === 1) {
            $supplier = $supplier_codes[0];
        } else {
            $supplier = 'X';
        }

        return $city . $district . $supplier;
    }

    /**
     * Set variation's logistics code using an array of supplier nodes as sources.
     */
    public function setVariationLogisticsCodeFromSupplierNodes($variation, array $supplier_nodes): void
    {
        if (!$variation->hasField('field_logistics_code')) {
            return;
        }

        // Variation must be saved to have an ID
        if (!$variation->id()) {
            return;
        }

        $body = $this->computeLogisticsCodeBodyFromSupplierNodes($supplier_nodes);
        $variation_id_padded = str_pad($variation->id(), 6, '0', STR_PAD_LEFT);
        // Prefix with 'V' for Variation
        $code = $body . 'V' . $variation_id_padded;
        $variation->set('field_logistics_code', $code);
        try {
            $variation->save();
        } catch (\Throwable $e) {
            $this->logger->error('Failed to save logistics code for variation @vid: @msg', ['@vid' => $variation->id(), '@msg' => $e->getMessage()]);
        }
    }

    /**
     * Set product's logistics code using an array of supplier nodes as sources.
     */
    public function setProductLogisticsCodeFromSupplierNodes($product, array $supplier_nodes): void
    {
        if (!$product->hasField('field_logistics_code')) {
            return;
        }

        // Product must be saved to have an ID
        if (!$product->id()) {
            return;
        }

        $body = $this->computeLogisticsCodeBodyFromSupplierNodes($supplier_nodes);
        $product_id_padded = str_pad($product->id(), 6, '0', STR_PAD_LEFT);
        // Prefix with 'P' for Product
        $code = $body . 'P' . $product_id_padded;
        $product->set('field_logistics_code', $code);
        try {
            $product->save();
        } catch (\Throwable $e) {
            $this->logger->error('Failed to save logistics code for product @nid: @msg', ['@nid' => $product->id(), '@msg' => $e->getMessage()]);
        }
    }


    public function getAttributeValueFromNode($node, $target_key)
    {
        if ($node->hasField('field_product_attributes')) {
            $values = $node->get('field_product_attributes')->getValue();
            foreach ($values as $item) {
                if (isset($item['key']) && $item['key'] === $target_key && isset($item['value'])) {
                    return $item['value'];
                }
            }
        }
        return NULL;
    }
    /**
     * Find existing commerce products that reference any of the given supplier nodes.
     *
     * @param array $nodes
     *   Array of supplier product nodes.
     *
     * @return array
     *   Array of commerce product entities that have variations referencing these suppliers.
     */
    protected function findExistingProducts(array $nodes): array
    {
        if (empty($nodes)) {
            return [];
        }

        $node_ids = array_map(function ($node) {
            return $node->id();
        }, $nodes);

        // Query variations with field_source_supplier_products referencing these nodes
        $variation_storage = $this->entityTypeManager->getStorage('commerce_product_variation');
        $query = $variation_storage->getQuery()
            ->condition('field_source_supplier_products', $node_ids, 'IN')
            // Access bypass intentional: Backend service operation without user context.
            ->accessCheck(FALSE);

        $variation_ids = $query->execute();

        if (empty($variation_ids)) {
            return [];
        }

        // Load variations and get parent products
        $variations = $variation_storage->loadMultiple($variation_ids);
        $product_ids = [];

        foreach ($variations as $variation) {
            if ($variation->hasField('product_id') && !$variation->get('product_id')->isEmpty()) {
                $product_id = $variation->get('product_id')->target_id;
                if ($product_id) {
                    $product_ids[$product_id] = $product_id;
                }
            }
        }

        if (empty($product_ids)) {
            return [];
        }

        // Load and return products
        $product_storage = $this->entityTypeManager->getStorage('commerce_product');
        return $product_storage->loadMultiple($product_ids);
    }

    // ========================================================================
    // PRICING CALCULATION METHODS
    // ========================================================================

    /**
     * Get price and stock data from a supplier product node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return array
     *   Array with keys: stock, cost, list, suggested (all nullable).
     */
    protected function getSupplierPriceData($node): array
    {
        $data = [
            'stock' => 0,
            'cost' => NULL,
            'list' => NULL,
            'suggested' => NULL,
        ];

        // Stock
        // Prefer using StockManager which knows the canonical supplier stock
        // field and semantics. Fall back to 'field_stock' if StockManager is
        // not available.
        if ($this->stockManager) {
            $data['stock'] = (int) $this->stockManager->getSupplierStock($node);
        } elseif ($node->hasField('field_stock') && !$node->get('field_stock')->isEmpty()) {
            $data['stock'] = (int) $node->get('field_stock')->value;
        }

        // Cost price
        if ($node->hasField('field_cost_price') && !$node->get('field_cost_price')->isEmpty()) {
            $data['cost'] = (float) $node->get('field_cost_price')->number;
        }

        // List price
        if ($node->hasField('field_list_price') && !$node->get('field_list_price')->isEmpty()) {
            $data['list'] = (float) $node->get('field_list_price')->number;
        }

        // Suggested price (selling price)
        if ($node->hasField('field_suggested_price') && !$node->get('field_suggested_price')->isEmpty()) {
            $data['suggested'] = (float) $node->get('field_suggested_price')->number;
        }

        return $data;
    }



    // ========================================================================
    // VARIATION MATCHING & UPDATE METHODS
    // ========================================================================



    /**
     * Update an existing variation with new supplier sources.
     *
     * Recalculates prices, updates stock, and logistics code.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation to update.
     * @param array $new_supplier_nodes
     *   New supplier nodes to add as sources.
     */
    public function updateVariationWithSuppliers($variation, array $new_supplier_nodes): void
    {
        $this->variationBuilder->updateVariationWithSuppliers($variation, $new_supplier_nodes);
    }

    /**
     * Recalculate product fields (title, body) from all linked supplier products.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The commerce product to update.
     */
    public function recalculateProductFields($product)
    {
        $all_supplier_nodes = [];
        $seen_ids = [];

        // Collect all supplier nodes from all variations
        foreach ($product->getVariations() as $variation) {
            if ($variation->hasField('field_source_supplier_products') && !$variation->get('field_source_supplier_products')->isEmpty()) {
                $referenced_entities = $variation->get('field_source_supplier_products')->referencedEntities();
                foreach ($referenced_entities as $node) {
                    if (!isset($seen_ids[$node->id()])) {
                        $all_supplier_nodes[] = $node;
                        $seen_ids[$node->id()] = TRUE;
                    }
                }
            }
        }

        if (empty($all_supplier_nodes)) {
            return;
        }

        // Update Title
        $title = $this->contentResolver->selectTextByStrategy($all_supplier_nodes, 'title', 'product_name_strategy');
        if ($title && $title !== $product->label()) {
            $product->setTitle($title);
        }

        // Update Description
        $description_value = $this->contentResolver->selectTextByStrategy($all_supplier_nodes, 'description', 'product_description_strategy');
        if ($description_value !== NULL && $product->hasField('body')) {
            $product->set('body', [
                'value' => $description_value,
                'format' => 'full_html',
            ]);
        }

        // Update Images
        $this->imageHandler->recalculateImages($product, $all_supplier_nodes);

        // Update Logistics Code
        $this->setProductLogisticsCodeFromSupplierNodes($product, $all_supplier_nodes);

        $product->save();
    }



    // ========================================================================
    // MAIN ORCHESTRATION METHODS
    // ========================================================================

    /**
     * Create a new commerce product from grouped supplier nodes.
     *
     * This is the new intelligent merge implementation.
     *
     * @param array $groups
     *   Grouped suppliers by attribute signature from groupSuppliersByAttributes().
     * @param array $all_nodes
     *   All supplier nodes (expanded).
     *
     * @return \Drupal\commerce_product\Entity\ProductInterface
     *   The created product.
     */
    protected function createNewProductFromGroups(array $groups, array $all_nodes)
    {
        // Find oldest node for product title/description
        $oldest_node = NULL;
        $oldest_timestamp = PHP_INT_MAX;

        foreach ($all_nodes as $node) {
            if ($node->hasField('created') && $node->get('created')->value < $oldest_timestamp) {
                $oldest_timestamp = $node->get('created')->value;
                $oldest_node = $node;
            }
        }

        if (!$oldest_node) {
            $oldest_node = reset($all_nodes);
        }

        // Get product title
        $title = $this->contentResolver->selectTextByStrategy($all_nodes, 'title', 'product_name_strategy');
        if (empty($title)) {
            $title = $oldest_node->label();
        }

        // Create product
        $product = Product::create([
            'type' => 'default',
            'title' => $title,
            'status' => TRUE, // Published
            'stores' => [$this->entityTypeManager->getStorage('commerce_store')->load(1)],
        ]);

        // Set description from oldest
        $description_value = $this->contentResolver->selectTextByStrategy($all_nodes, 'description', 'product_description_strategy');
        if ($description_value !== NULL && $product->hasField('body')) {
            $product->set('body', [
                'value' => $description_value,
                'format' => 'full_html',
            ]);
        }

        $product->save();

        // Create images from all suppliers
        $this->imageHandler->addImagesToProduct($product, $all_nodes);

        // Create product-level taxonomies from ALL attributes
        $this->createProductTaxonomiesFromAllAttributes($product, $all_nodes);

        // Create variations from groups
        foreach ($groups as $signature => $group) {
            $this->variationBuilder->createVariationFromGroup($product, $group);
        }

        // Reload and save product to update references
        $product->save();

        return $product;
    }

    /**
     * Update an existing product with new grouped supplier nodes.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The existing product.
     * @param array $groups
     *   Grouped suppliers by attribute signature.
     */
    protected function updateExistingProductWithGroups($product, array $groups): void
    {
        // Get existing variations
        $existing_variations = $product->getVariations();

        foreach ($groups as $signature => $group) {
            $matched = FALSE;
            $normalized_attrs = $group['normalized_attrs'];

            // Try to match existing variation
            foreach ($existing_variations as $variation) {
                if ($this->attributeResolver->matchVariationByAttributes($variation, $normalized_attrs)) {
                    // Match found - add suppliers to this variation
                    $this->updateVariationWithSuppliers($variation, $group['nodes']);
                    $matched = TRUE;
                    break;
                }
            }

            // No match - create new variation
            if (!$matched) {
                $this->variationBuilder->createVariationFromGroup($product, $group);
            }
        }

        $product->save();
    }





    /**
     * Create product-level taxonomies from ALL attributes across all suppliers.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product.
     * @param array $nodes
     *   All supplier nodes.
     */
    protected function createProductTaxonomiesFromAllAttributes($product, array $nodes): void
    {
        $all_attributes = [];

        // Collect all unique attribute label/value pairs
        foreach ($nodes as $node) {
            $attrs = $this->attributeResolver->extractAttributes($node);
            foreach ($attrs['original'] as $label => $value) {
                if (!isset($all_attributes[$label])) {
                    $all_attributes[$label] = [];
                }
                $all_attributes[$label][$value] = $value;
            }
        }

        // Create term references for each attribute
        foreach ($all_attributes as $label => $values) {
            $vocabulary_name = 'product_' . strtolower($this->attributeResolver->normalizeAttributeLabel($label));
            $term_ids = [];

            foreach ($values as $value) {
                // Use TaxonomyManager to create vocabulary and terms
                $resolved = $this->taxonomyManager->resolveSuggestedValues($vocabulary_name, $value);
                foreach ($resolved as $r) {
                    if (!empty($r['target_id'])) {
                        $term_ids[] = (int) $r['target_id'];
                    }
                }
            }

            if (!empty($term_ids)) {
                $field_name = 'field_tax_' . strtolower($this->attributeResolver->normalizeAttributeLabel($label));
                // Note: Field might not exist yet - this is a simplified version
                // In production, you'd need to ensure field exists first
                if ($product->hasField($field_name)) {
                    $product->set($field_name, array_map(function ($tid) {
                        return ['target_id' => $tid];
                    }, $term_ids));
                }
            }
        }
    }

    /**
     * Entry point for VBO action - merge and create product from supplier nodes.
     *
     * Returns array with status and info for user feedback.
     *
     * @param array $supplier_nodes
     *   Selected supplier product nodes.
     *
     * @return array
     *   Result array with 'status', 'product', 'message', 'existing_products'.
     */
    public function mergeCreateProduct(array $supplier_nodes): array
    {
        if (empty($supplier_nodes)) {
            return [
                'status' => 'error',
                'message' => 'No supplier products selected.',
            ];
        }

        // Phase 1: Expansion
        $expanded_nodes = $this->expandVariations($supplier_nodes);

        // Phase 2: Grouping
        $groups = $this->attributeResolver->groupSuppliersByAttributes($expanded_nodes);

        // Phase 3: Check for existing products
        $existing_products = $this->findExistingProducts($expanded_nodes);

        if (count($existing_products) > 1) {
            // Multiple products - need user confirmation
            return [
                'status' => 'confirm',
                'message' => 'Multiple existing products found. User must choose.',
                'existing_products' => $existing_products,
                'groups' => $groups,
                'expanded_nodes' => $expanded_nodes,
            ];
        } elseif (count($existing_products) === 1) {
            // Update existing product
            $product = reset($existing_products);
            $this->updateExistingProductWithGroups($product, $groups);

            return [
                'status' => 'success',
                'product' => $product,
                'message' => sprintf('Updated product "%s" with %d supplier products.', $product->label(), count($expanded_nodes)),
            ];
        } else {
            // Create new product
            $product = $this->createNewProductFromGroups($groups, $expanded_nodes);

            return [
                'status' => 'success',
                'product' => $product,
                'message' => sprintf('Created product "%s" with %d variations from %d supplier products.', $product->label(), count($groups), count($expanded_nodes)),
            ];
        }
    }
}
