<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for calculating prices from supplier product nodes.
 *
 * Handles cost price, list price, selling price, and tax rate calculations
 * using configurable strategies (minimum, maximum, average, weighted).
 */
class PriceCalculator
{

    /**
     * The stock manager service.
     *
     * @var \Drupal\sogan_commerce_product\Service\StockManager|null
     */
    protected ?StockManager $stockManager;

    /**
     * Merge policy configuration.
     *
     * @var array
     */
    protected array $mergePolicy;

    /**
     * Constructs a PriceCalculator object.
     *
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\sogan_commerce_product\Service\StockManager $stock_manager
     *   The stock manager service.
     */
    public function __construct(
        ConfigFactoryInterface $config_factory,
        StockManager $stock_manager
    ) {
        $config = $config_factory->get('sogan_commerce_product.settings');
        $this->mergePolicy = $config->get('merge_policy') ?? [];
        $this->stockManager = $stock_manager;
    }

    /**
     * Get price and stock data from a supplier product node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return array
     *   Array with keys: stock, cost, list, suggested (all nullable).
     */
    public function getSupplierPriceData($node): array
    {
        $data = [
            'stock' => 0,
            'cost' => NULL,
            'list' => NULL,
            'suggested' => NULL,
        ];

        // Stock - prefer StockManager if available
        if ($this->stockManager) {
            $data['stock'] = (int) $this->stockManager->getSupplierStock($node);
        } elseif ($node->hasField('field_stock') && !$node->get('field_stock')->isEmpty()) {
            $data['stock'] = (int) $node->get('field_stock')->value;
        }

        // Cost price
        if ($node->hasField('field_cost_price') && !$node->get('field_cost_price')->isEmpty()) {
            $data['cost'] = (float) $node->get('field_cost_price')->number;
        }

        // List price
        if ($node->hasField('field_list_price') && !$node->get('field_list_price')->isEmpty()) {
            $data['list'] = (float) $node->get('field_list_price')->number;
        }

        // Suggested price (selling price)
        if ($node->hasField('field_suggested_price') && !$node->get('field_suggested_price')->isEmpty()) {
            $data['suggested'] = (float) $node->get('field_suggested_price')->number;
        }

        return $data;
    }

    /**
     * Calculate cost price from multiple suppliers.
     *
     * @param array $supplier_nodes
     *   Array of supplier product nodes.
     *
     * @return float|null
     *   Cost price or NULL if none available.
     */
    public function calculateCostPrice(array $supplier_nodes): ?float
    {
        return $this->calculatePriceByStrategy($supplier_nodes, 'cost', 'cost_price_strategy');
    }

    /**
     * Calculate list price from multiple suppliers.
     *
     * @param array $supplier_nodes
     *   Array of supplier product nodes.
     *
     * @return float|null
     *   List price or NULL if none available.
     */
    public function calculateListPrice(array $supplier_nodes): ?float
    {
        return $this->calculatePriceByStrategy($supplier_nodes, 'list', 'list_price_strategy');
    }

    /**
     * Calculate selling price from multiple suppliers.
     *
     * @param array $supplier_nodes
     *   Array of supplier product nodes.
     *
     * @return float|null
     *   Selling price (2 decimals) or NULL if none available.
     */
    public function calculateSellingPrice(array $supplier_nodes): ?float
    {
        return $this->calculatePriceByStrategy($supplier_nodes, 'suggested', 'price_strategy');
    }

    /**
     * Calculate tax rate ID from multiple suppliers.
     *
     * Uses the highest VAT rate among all supplier products to determine
     * the appropriate tax rate ID for the variation.
     *
     * @param array $supplier_nodes
     *   Array of supplier product nodes.
     *
     * @return string|null
     *   Tax rate ID ('standard', 'reduced_high', 'reduced_low') or NULL.
     */
    public function calculateTaxRateFromSuppliers(array $supplier_nodes): ?string
    {
        if (empty($supplier_nodes)) {
            return NULL;
        }

        $max_vat_rate = NULL;

        foreach ($supplier_nodes as $node) {
            if (!$node->hasField('field_vat_rate') || $node->get('field_vat_rate')->isEmpty()) {
                continue;
            }

            $vat_rate = (float) $node->get('field_vat_rate')->value;

            if ($max_vat_rate === NULL || $vat_rate > $max_vat_rate) {
                $max_vat_rate = $vat_rate;
            }
        }

        if ($max_vat_rate === NULL) {
            return NULL;
        }

        // Map VAT rate percentages to tax rate IDs based on Turkey VAT rates.
        // Standard: 20%, Reduced High: 10%, Reduced Low: 1%
        if ($max_vat_rate >= 15) {
            return 'standard';
        } elseif ($max_vat_rate >= 5) {
            return 'reduced_high';
        } else {
            return 'reduced_low';
        }
    }

    /**
     * Calculate price from supplier nodes based on configured strategy.
     *
     * @param array $nodes
     *   Array of supplier product nodes.
     * @param string $price_field
     *   Price field name ('cost', 'list', or 'suggested').
     * @param string $strategy_key
     *   Key in merge policy config.
     *
     * @return float|null
     *   Calculated price or NULL.
     */
    public function calculatePriceByStrategy(array $nodes, string $price_field, string $strategy_key): ?float
    {
        if (empty($nodes)) {
            return NULL;
        }

        $strategy = $this->mergePolicy[$strategy_key] ?? 'minimum';
        $only_instock = $this->mergePolicy['only_instock'] ?? FALSE;
        $prices = [];

        foreach ($nodes as $node) {
            $data = $this->getSupplierPriceData($node);

            // Skip if price not available or zero
            if ($data[$price_field] === NULL || (float) $data[$price_field] <= 0) {
                continue;
            }

            // Apply in-stock filter if enabled
            if ($only_instock && $data['stock'] <= 0) {
                continue;
            }

            $prices[] = [
                'price' => $data[$price_field],
                'stock' => $data['stock'],
            ];
        }

        if (empty($prices)) {
            return NULL;
        }

        // Apply strategy
        switch ($strategy) {
            case 'minimum':
                return round(min(array_column($prices, 'price')), 2);

            case 'maximum':
                return round(max(array_column($prices, 'price')), 2);

            case 'average':
                $sum = array_sum(array_column($prices, 'price'));
                return round($sum / count($prices), 2);

            case 'weighted':
                $total_stock = array_sum(array_column($prices, 'stock'));
                if ($total_stock <= 0) {
                    // Fallback to simple average if no stock
                    $sum = array_sum(array_column($prices, 'price'));
                    return round($sum / count($prices), 2);
                }
                $weighted_sum = 0;
                foreach ($prices as $item) {
                    $weighted_sum += ($item['stock'] * $item['price']);
                }
                return round($weighted_sum / $total_stock, 2);

            default:
                return round($prices[0]['price'], 2);
        }
    }
}
