<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\TermInterface;

/**
 * Service for generating logistics codes for products and variations.
 *
 * Logistics codes are formatted as: {city_code}{district_code}{supplier_code}{type}{id}
 * Example: 34KCV000012 (city 34, district K, supplier C, Variation, ID 12)
 */
class LogisticsCodeGenerator
{

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a LogisticsCodeGenerator object.
     *
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     */
    public function __construct(LoggerChannelFactoryInterface $logger_factory)
    {
        $this->logger = $logger_factory->get('sogan_commerce_product');
    }

    /**
     * Compute the logistics code body string from an array of supplier nodes.
     *
     * @param array $supplier_nodes
     *   Array of supplier product node entities.
     *
     * @return string
     *   The logistics code body (e.g., '34KC' - city(2)+district(1)+supplier(1)).
     */
    public function computeLogisticsCodeBodyFromSupplierNodes(array $supplier_nodes): string
    {
        $city_codes = [];
        $district_codes = [];
        $supplier_codes = [];

        foreach ($supplier_nodes as $n) {
            $term = $this->extractSupplierTerm($n);
            if (!$term) {
                continue;
            }

            if ($term->hasField('field_city_code') && !$term->get('field_city_code')->isEmpty()) {
                $city_codes[] = (string) $term->get('field_city_code')->value;
            }
            if ($term->hasField('field_district_code') && !$term->get('field_district_code')->isEmpty()) {
                $district_codes[] = (string) $term->get('field_district_code')->value;
            }
            if ($term->hasField('field_supplier_code') && !$term->get('field_supplier_code')->isEmpty()) {
                $supplier_codes[] = (string) $term->get('field_supplier_code')->value;
            }
        }

        // Deduplicate and filter empties
        $city_codes = array_values(array_unique(array_filter($city_codes)));
        $district_codes = array_values(array_unique(array_filter($district_codes)));
        $supplier_codes = array_values(array_unique(array_filter($supplier_codes)));

        // City: if >1 then '99', if 1 then value, else 'XX'
        if (count($city_codes) > 1) {
            $city = '99';
        } elseif (count($city_codes) === 1) {
            $city = $city_codes[0];
        } else {
            $city = 'XX';
        }

        // District: >1 -> 'X', 1 -> value, else 'X'
        if (count($district_codes) > 1) {
            $district = 'X';
        } elseif (count($district_codes) === 1) {
            $district = $district_codes[0];
        } else {
            $district = 'X';
        }

        // Supplier: >1 -> 'X', 1 -> value, else 'X'
        if (count($supplier_codes) > 1) {
            $supplier = 'X';
        } elseif (count($supplier_codes) === 1) {
            $supplier = $supplier_codes[0];
        } else {
            $supplier = 'X';
        }

        return $city . $district . $supplier;
    }

    /**
     * Set variation's logistics code using an array of supplier nodes.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation entity.
     * @param array $supplier_nodes
     *   Array of supplier product node entities.
     */
    public function setVariationLogisticsCodeFromSupplierNodes(ProductVariationInterface $variation, array $supplier_nodes): void
    {
        if (!$variation->hasField('field_logistics_code')) {
            return;
        }

        // Variation must be saved to have an ID
        if (!$variation->id()) {
            return;
        }

        $body = $this->computeLogisticsCodeBodyFromSupplierNodes($supplier_nodes);
        $variation_id_padded = str_pad($variation->id(), 6, '0', STR_PAD_LEFT);
        // Prefix with 'V' for Variation
        $code = $body . 'V' . $variation_id_padded;
        $variation->set('field_logistics_code', $code);

        try {
            $variation->save();
        } catch (\Throwable $e) {
            $this->logger->error(
                'Failed to save logistics code for variation @vid: @msg',
                ['@vid' => $variation->id(), '@msg' => $e->getMessage()]
            );
        }
    }

    /**
     * Set product's logistics code using an array of supplier nodes.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product entity.
     * @param array $supplier_nodes
     *   Array of supplier product node entities.
     */
    public function setProductLogisticsCodeFromSupplierNodes(ProductInterface $product, array $supplier_nodes): void
    {
        if (!$product->hasField('field_logistics_code')) {
            return;
        }

        // Product must be saved to have an ID
        if (!$product->id()) {
            return;
        }

        $body = $this->computeLogisticsCodeBodyFromSupplierNodes($supplier_nodes);
        $product_id_padded = str_pad($product->id(), 6, '0', STR_PAD_LEFT);
        // Prefix with 'P' for Product
        $code = $body . 'P' . $product_id_padded;
        $product->set('field_logistics_code', $code);

        try {
            $product->save();
        } catch (\Throwable $e) {
            $this->logger->error(
                'Failed to save logistics code for product @nid: @msg',
                ['@nid' => $product->id(), '@msg' => $e->getMessage()]
            );
        }
    }

    /**
     * Set logistics code using a supplier term or node (backwards compatible wrapper).
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation entity.
     * @param \Drupal\taxonomy\TermInterface|\Drupal\node\NodeInterface|mixed $supplier_term_or_node
     *   A supplier term or supplier product node.
     */
    public function setVariationLogisticsCode(ProductVariationInterface $variation, $supplier_term_or_node): void
    {
        if (!$variation->hasField('field_logistics_code')) {
            return;
        }

        $nodes = [];
        if ($supplier_term_or_node instanceof NodeInterface) {
            $nodes[] = $supplier_term_or_node;
        } elseif ($supplier_term_or_node instanceof TermInterface) {
            // Wrap a term into a synthetic structure
            $fake_node = new \stdClass();
            $fake_node->field_supplier = (object) ['entity' => $supplier_term_or_node];
            $nodes[] = $fake_node;
        }

        $this->setVariationLogisticsCodeFromSupplierNodes($variation, $nodes);
    }

    /**
     * Extract supplier term from a node or wrapper object.
     *
     * @param mixed $n
     *   A NodeInterface or stdClass wrapper.
     *
     * @return \Drupal\taxonomy\TermInterface|null
     *   The supplier term or NULL.
     */
    protected function extractSupplierTerm($n): ?TermInterface
    {
        if ($n instanceof NodeInterface && $n->hasField('field_supplier') && !$n->get('field_supplier')->isEmpty()) {
            $term = $n->get('field_supplier')->entity;
            return $term instanceof TermInterface ? $term : NULL;
        }

        if (is_object($n) && isset($n->field_supplier->entity)) {
            $term = $n->field_supplier->entity;
            return $term instanceof TermInterface ? $term : NULL;
        }

        return NULL;
    }
}
