<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_stock\StockLocationInterface;

/**
 * Service to manage Local Stock operations.
 *
 * Local Stock acts as an intermediate buffer for order transactions,
 * isolating supplier stock (synced from feeds) from sales deductions.
 *
 * The buffer stock location is configured via the Local Stock Settings form.
 */
class LocalStockService
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The stock manager.
   *
   * @var \Drupal\sogan_commerce_product\Service\StockManager
   */
  protected $stockManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Cached Local Stock location.
   *
   * @var \Drupal\commerce_stock\StockLocationInterface|null
   */
  protected $cachedLocation;

  /**
   * Constructs a new LocalStockService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\sogan_commerce_product\Service\StockManager $stock_manager
   *   The stock manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    StockManager $stock_manager,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->stockManager = $stock_manager;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('sogan_commerce_product');
  }

  /**
   * Get the configured buffer stock location.
   *
   * The location is configured via admin/commerce/config/synchronization/local-stock
   *
   * @return \Drupal\commerce_stock\StockLocationInterface|null
   *   The buffer stock location entity, or NULL if not configured.
   */
  public function getLocalStockLocation(): ?StockLocationInterface
  {
    // Return cached location if available.
    if ($this->cachedLocation) {
      return $this->cachedLocation;
    }

    // Get the configured buffer location ID.
    $config = $this->configFactory->get('sogan_commerce_product.local_stock_settings');
    $location_id = $config->get('buffer_location_id');

    if (empty($location_id)) {
      $this->logger->warning(
        'Local Stock: Buffer stock location not configured. Please configure at /admin/commerce/config/synchronization/local-stock'
      );
      return NULL;
    }

    $location = $this->entityTypeManager->getStorage('commerce_stock_location')->load($location_id);

    if (!$location) {
      $this->logger->error(
        'Local Stock: Configured buffer location ID @id does not exist.',
        ['@id' => $location_id]
      );
      return NULL;
    }

    $this->cachedLocation = $location;
    return $this->cachedLocation;
  }


  /**
   * Get the current Local Stock level for a variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   *
   * @return float
   *   The current stock level (can be negative).
   */
  public function getLocalStockLevel(ProductVariationInterface $variation): float
  {
    $location = $this->getLocalStockLocation();
    if (!$location) {
      return 0.0;
    }

    return $this->stockManager->getStockLevel($variation, $location);
  }

  /**
   * Deduct stock from Local Stock for an order.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param float $quantity
   *   The quantity to deduct (positive number).
   * @param string $message
   *   Optional transaction message.
   *
   * @return int|false
   *   The transaction ID or FALSE on failure.
   */
  public function deductStock(ProductVariationInterface $variation, float $quantity, string $message = ''): int|false
  {
    $location = $this->getLocalStockLocation();
    if (!$location) {
      $this->logger->error('Cannot deduct stock: Local Stock location not found');
      return FALSE;
    }

    // Deduct is a negative transaction.
    $transaction_qty = -1 * abs($quantity);

    return $this->stockManager->createTransaction(
      $variation,
      $location,
      $transaction_qty,
      \Drupal\commerce_stock\StockTransactionsInterface::STOCK_SALE,
      $message ?: 'Order stock deduction from Local Stock'
    );
  }

  /**
   * Return stock to Local Stock (e.g., order cancellation).
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param float $quantity
   *   The quantity to return (positive number).
   * @param string $message
   *   Optional transaction message.
   *
   * @return int|false
   *   The transaction ID or FALSE on failure.
   */
  public function returnStock(ProductVariationInterface $variation, float $quantity, string $message = ''): int|false
  {
    $location = $this->getLocalStockLocation();
    if (!$location) {
      $this->logger->error('Cannot return stock: Local Stock location not found');
      return FALSE;
    }

    return $this->stockManager->createTransaction(
      $variation,
      $location,
      abs($quantity),
      \Drupal\commerce_stock\StockTransactionsInterface::STOCK_RETURN,
      $message ?: 'Order stock return to Local Stock'
    );
  }

  /**
   * Reset negative Local Stock to zero after shipment.
   *
   * If Local Stock is negative, it means items were sourced from suppliers.
   * After shipment, reset to 0 since the sourcing is complete.
   * If Local Stock is positive, keep it as-is (items came from local inventory).
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   * @param string $message
   *   Optional transaction message.
   *
   * @return int|false|null
   *   The transaction ID, FALSE on failure, or NULL if no reset needed.
   */
  public function resetNegativeStockAfterShipment(ProductVariationInterface $variation, string $message = ''): int|false|null
  {
    $location = $this->getLocalStockLocation();
    if (!$location) {
      return FALSE;
    }

    $current_level = $this->stockManager->getStockLevel($variation, $location);

    // Only reset if negative.
    if ($current_level >= 0) {
      return NULL; // No reset needed.
    }

    // Create a transaction to bring it back to 0.
    $adjustment = abs($current_level);

    return $this->stockManager->createTransaction(
      $variation,
      $location,
      $adjustment,
      \Drupal\commerce_stock\StockTransactionsInterface::STOCK_IN,
      $message ?: 'Reset Local Stock after shipment (sourced from suppliers)'
    );
  }

  /**
   * Check if a location is the Local Stock location.
   *
   * @param \Drupal\commerce_stock\StockLocationInterface $location
   *   The location to check.
   *
   * @return bool
   *   TRUE if this is the Local Stock location.
   */
  public function isLocalStockLocation(StockLocationInterface $location): bool
  {
    $local = $this->getLocalStockLocation();
    if (!$local) {
      return FALSE;
    }
    return $location->id() === $local->id();
  }
}
