<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\node\NodeInterface;

/**
 * Service for handling product images from supplier products.
 */
class ImageHandler
{

    /**
     * The remote media manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\RemoteMediaManager
     */
    protected $remoteMediaManager;

    /**
     * The config.
     *
     * @var \Drupal\Core\Config\ImmutableConfig
     */
    protected $config;

    /**
     * Constructs an ImageHandler object.
     */
    public function __construct(
        RemoteMediaManager $remote_media_manager,
        ConfigFactoryInterface $config_factory
    ) {
        $this->remoteMediaManager = $remote_media_manager;
        $this->config = $config_factory->get('sogan_commerce_product.settings');
    }

    /**
     * Add images from all supplier nodes to product.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product.
     * @param array $nodes
     *   Supplier nodes.
     */
    public function addImagesToProduct(ProductInterface $product, array $nodes): void
    {
        $media_items = [];

        foreach ($nodes as $node) {
            if (!$node instanceof NodeInterface) {
                continue;
            }
            if ($node->hasField('field_supplier_image_urls') && !$node->get('field_supplier_image_urls')->isEmpty()) {
                $image_values = $node->get('field_supplier_image_urls')->getValue();

                foreach ($image_values as $item) {
                    $uri = $item['uri'] ?? ($item['value'] ?? NULL);
                    if (empty($uri)) {
                        continue;
                    }

                    $context = [
                        'supplier_node' => $node,
                        'product' => $product,
                    ];

                    $media_id = $this->remoteMediaManager->createMediaFromRemoteUrl($uri, $product->label(), $context);
                    if ($media_id) {
                        $media_items[] = ['target_id' => $media_id];
                    }
                }
            }
        }

        if (!empty($media_items) && $product->hasField('field_product_images')) {
            $product->set('field_product_images', $media_items);
        }
    }

    /**
     * Recalculate images for a product based on strategy.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product.
     * @param array $all_supplier_nodes
     *   All supplier nodes.
     */
    public function recalculateImages(ProductInterface $product, array $all_supplier_nodes): void
    {
        if (!$product->hasField('field_product_images')) {
            return;
        }

        $image_strategy = $this->config->get('merge_policy.product_image_strategy') ?? 'combine';
        $nodes_for_images = $all_supplier_nodes;

        if ($image_strategy === 'most_images') {
            $nodes_for_images = $this->selectNodeWithMostImages($all_supplier_nodes);
        }

        $media_items = $this->collectMediaItems($nodes_for_images, $product);

        // Deduplicate media items by target_id
        $unique_media_items = $this->deduplicateMediaItems($media_items);

        // Update the product images
        $product->set('field_product_images', $unique_media_items);
    }

    /**
     * Select the node with the most images.
     *
     * @param array $nodes
     *   Supplier nodes.
     *
     * @return array
     *   Array containing only the node with most images, or empty array.
     */
    protected function selectNodeWithMostImages(array $nodes): array
    {
        $best_node = NULL;
        $max_count = -1;

        foreach ($nodes as $node) {
            if (!$node instanceof NodeInterface) {
                continue;
            }
            $count = 0;
            if ($node->hasField('field_images') && !$node->get('field_images')->isEmpty()) {
                $count += count($node->get('field_images')->getValue());
            }
            if ($node->hasField('field_remote_images_url') && !$node->get('field_remote_images_url')->isEmpty()) {
                $count += count($node->get('field_remote_images_url')->getValue());
            }

            if ($count > $max_count) {
                $max_count = $count;
                $best_node = $node;
            }
        }

        return $best_node ? [$best_node] : [];
    }

    /**
     * Collect media items from supplier nodes.
     *
     * @param array $nodes
     *   Supplier nodes.
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The product.
     *
     * @return array
     *   Array of media item references.
     */
    protected function collectMediaItems(array $nodes, ProductInterface $product): array
    {
        $media_items = [];

        foreach ($nodes as $node) {
            if (!$node instanceof NodeInterface) {
                continue;
            }

            // 1. Existing Media
            if ($node->hasField('field_images') && !$node->get('field_images')->isEmpty()) {
                $items = $node->get('field_images')->getValue();
                foreach ($items as $item) {
                    $media_items[] = ['target_id' => $item['target_id']];
                }
            }

            // 2. Remote Images
            if ($node->hasField('field_supplier_image_urls') && !$node->get('field_supplier_image_urls')->isEmpty()) {
                $image_values = $node->get('field_supplier_image_urls')->getValue();

                foreach ($image_values as $item) {
                    $uri = $item['uri'] ?? ($item['value'] ?? NULL);
                    if (empty($uri)) {
                        continue;
                    }

                    $context = [
                        'supplier_node' => $node,
                        'product' => $product,
                    ];

                    $media_id = $this->remoteMediaManager->createMediaFromRemoteUrl($uri, $product->label(), $context);
                    if ($media_id) {
                        $media_items[] = ['target_id' => $media_id];
                    }
                }
            }
        }

        return $media_items;
    }

    /**
     * Deduplicate media items by target_id.
     *
     * @param array $media_items
     *   Media items to deduplicate.
     *
     * @return array
     *   Deduplicated media items.
     */
    protected function deduplicateMediaItems(array $media_items): array
    {
        $unique_items = [];
        $seen_ids = [];

        foreach ($media_items as $item) {
            if (!isset($seen_ids[$item['target_id']])) {
                $unique_items[] = $item;
                $seen_ids[$item['target_id']] = TRUE;
            }
        }

        return $unique_items;
    }
}
